package ru.yandex.qe.dispenser.domain;

import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.builder.CompareToBuilder;
import org.jetbrains.annotations.NotNull;

import ru.yandex.qe.dispenser.api.DtoBuilder;
import ru.yandex.qe.dispenser.api.v1.DiSegmentation;
import ru.yandex.qe.dispenser.domain.index.NormalizedPrimaryKeyBase;

/**
 * Сегментация - способ разделить {@link Quota квот} по некоторому принципу.
 * Объединяет несколько {@link Segment сегментов}.
 * Для привязки квотируемого {@link Resource ресурса} используется класс {@link ResourceSegmentation}.
 *
 * @see Segment
 * @see ResourceSegmentation
 */
@ParametersAreNonnullByDefault
public final class Segmentation extends NormalizedPrimaryKeyBase<Segmentation.Key> {

    private final String name;
    private final String description;
    private final int priority;

    public Segmentation(final Builder builder) {
        super(new Key(builder.key));
        name = builder.name;
        description = builder.description;
        priority = builder.priority;
    }

    public String getName() {
        return name;
    }

    public String getDescription() {
        return description;
    }

    public int getPriority() {
        return priority;
    }

    public DiSegmentation toView() {
        return new DiSegmentation.Builder(getKey().getPublicKey())
                .withName(name)
                .withDescription(description)
                .withPriority(priority)
                .build();
    }

    public static final class Key implements Comparable<Key> {

        private final String key;

        public Key(final String publicKey) {
            this.key = publicKey;
        }

        @NotNull
        public String getPublicKey() {
            return key;
        }

        @Override
        public int compareTo(final Segmentation.Key key) {
            return new CompareToBuilder()
                    .append(getPublicKey(), key.getPublicKey())
                    .build();
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final Key key1 = (Key) o;
            return Objects.equals(key, key1.key);
        }

        @Override
        public int hashCode() {
            return Objects.hash(key);
        }

        @NotNull
        @Override
        public String toString() {
            return "Key{key='" + key + "'}";
        }
    }

    public static final class Builder implements DtoBuilder<Segmentation> {

        private final String key;
        private String name;
        private String description;
        private int priority;

        public Builder(final String key) {
            this.key = key;
        }

        public Builder name(final String name) {
            this.name = name;
            return this;
        }

        public Builder description(final String description) {
            this.description = description;
            return this;
        }

        public Builder priority(final int priority) {
            this.priority = priority;
            return this;
        }

        @NotNull
        @Override
        public Segmentation build() {
            return new Segmentation(this);
        }
    }
}
