package ru.yandex.qe.dispenser.domain.aspect;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.IntStream;

import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.BeanParam;
import javax.ws.rs.HeaderParam;
import javax.ws.rs.PathParam;
import javax.ws.rs.QueryParam;

import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.reflect.MethodSignature;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import ru.yandex.qe.dispenser.domain.util.StreamUtils;
import ru.yandex.qe.dispenser.domain.util.ValidationUtils;

public abstract class AspectBase {

    @NotNull
    protected String getHttpQueryParam(@NotNull final String name) {
        final HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.currentRequestAttributes()).getRequest();
        return ValidationUtils.requireNonNull(request.getParameter(name), "No cgi param '" + name + "' in request!");

    }

    @NotNull
    protected Object getQueryParam(@NotNull final JoinPoint jp, @NotNull final String paramName) {
        return ValidationUtils.requireNonNull(getQueryParams(jp).get(paramName), "No cgi param '" + paramName + "' in request!");
    }

    @NotNull
    protected String getStringQueryParam(@NotNull final JoinPoint jp, @NotNull final String paramName) {
        return (String) getQueryParam(jp, paramName);
    }

    @NotNull
    protected Map<String, Object> getQueryParams(@NotNull final JoinPoint jp) {
        return getParams(jp, QueryParam.class, QueryParam::value);
    }

    @NotNull
    protected Object getHeaderParam(@NotNull final JoinPoint jp, @NotNull final String paramName) {
        return ValidationUtils.requireNonNull(getHeaderParams(jp).get(paramName), "No header '" + paramName + "' in request!");
    }

    @NotNull
    protected String getStringHeaderParam(@NotNull final JoinPoint jp, @NotNull final String paramName) {
        return (String) getHeaderParam(jp, paramName);
    }

    @NotNull
    protected Map<String, Object> getHeaderParams(@NotNull final JoinPoint jp) {
        return getParams(jp, HeaderParam.class, HeaderParam::value);
    }

    @NotNull
    protected Object getPathParam(@NotNull final JoinPoint jp, @NotNull final String paramName) {
        return ValidationUtils.requireNonNull(getPathParams(jp).get(paramName), "No path param '" + paramName + "' in request!");
    }

    @NotNull
    protected String getStringPathParam(@NotNull final JoinPoint jp, @NotNull final String paramName) {
        return (String) getPathParam(jp, paramName);
    }

    @NotNull
    protected Map<String, Object> getPathParams(@NotNull final JoinPoint jp) {
        return getParams(jp, PathParam.class, PathParam::value);
    }

    @Nullable
    protected <T> T getFirstInstance(@NotNull final JoinPoint jp, @NotNull final Class<? extends T> argClass) {
        return StreamUtils.instances(jp.getArgs(), argClass).findFirst().orElse(null);
    }

    @NotNull
    protected <T extends Annotation> Map<String, Object> getParams(@NotNull final JoinPoint jp,
                                                                   @NotNull final Class<T> annotationClass,
                                                                   @NotNull final Function<? super T, String> paramName) {
        final Method method = ((MethodSignature) jp.getSignature()).getMethod();
        final Map<String, Object> params = new HashMap<>();
        getAnnotatedParameters(method, annotationClass)
                .forEach((i, queryParam) -> params.put(paramName.apply(queryParam), jp.getArgs()[i]));
        getAnnotatedParameters(method, BeanParam.class)
                .forEach((i, beanParam) -> {
                    final Object arg = jp.getArgs()[i];
                    for (final Field field : arg.getClass().getDeclaredFields()) {
                        final T annotation = field.getAnnotation(annotationClass);
                        if (annotation != null) {
                            field.setAccessible(true);
                            try {
                                params.put(paramName.apply(annotation), field.get(arg));
                            } catch (IllegalAccessException ignore) {
                            }
                        }
                    }
                });
        return params;
    }

    @NotNull
    protected <T extends Annotation> Map<Integer, T> getAnnotatedParameters(@NotNull final Method method,
                                                                            @NotNull final Class<? extends T> annotationClass) {
        final Annotation[][] annotations = method.getParameterAnnotations();
        final Map<Integer, T> index2annotation = new HashMap<>();
        IntStream.range(0, annotations.length)
                .forEach(i -> StreamUtils.instances(annotations[i], annotationClass)
                        .findFirst()
                        .ifPresent(annotation -> index2annotation.put(i, annotation)));
        return index2annotation;
    }
}
