package ru.yandex.qe.dispenser.domain.base_resources;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

public final class BaseResource {

    private final long id;
    private final String key;
    private final String name;
    private final long baseResourceTypeId;
    private final Set<Long> segmentIds;

    private BaseResource(long id, String key, String name, long baseResourceTypeId, Set<Long> segmentIds) {
        this.id = id;
        this.key = key;
        this.name = name;
        this.baseResourceTypeId = baseResourceTypeId;
        this.segmentIds = segmentIds;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(BaseResource value) {
        return new Builder(value);
    }

    public long getId() {
        return id;
    }

    public String getKey() {
        return key;
    }

    public String getName() {
        return name;
    }

    public long getBaseResourceTypeId() {
        return baseResourceTypeId;
    }

    public Set<Long> getSegmentIds() {
        return segmentIds;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        BaseResource that = (BaseResource) o;
        return id == that.id &&
                baseResourceTypeId == that.baseResourceTypeId &&
                Objects.equals(key, that.key) &&
                Objects.equals(name, that.name) &&
                Objects.equals(segmentIds, that.segmentIds);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, key, name, baseResourceTypeId, segmentIds);
    }

    @Override
    public String toString() {
        return "BaseResource{" +
                "id=" + id +
                ", key='" + key + '\'' +
                ", name='" + name + '\'' +
                ", baseResourceTypeId=" + baseResourceTypeId +
                ", segmentIds=" + segmentIds +
                '}';
    }

    public static final class Builder {

        private final Set<Long> segmentIds = new HashSet<>();

        private Long id;
        private String key;
        private String name;
        private Long baseResourceTypeId;

        private Builder() {
        }

        private Builder(BaseResource value) {
            this.id = value.id;
            this.key = value.key;
            this.name = value.name;
            this.baseResourceTypeId = value.baseResourceTypeId;
            this.segmentIds.addAll(value.segmentIds != null ? value.segmentIds : Set.of());
        }

        public Builder id(long id) {
            this.id = id;
            return this;
        }

        public Builder key(String key) {
            this.key = key;
            return this;
        }

        public Builder name(String name) {
            this.name = name;
            return this;
        }

        public Builder baseResourceTypeId(long baseResourceTypeId) {
            this.baseResourceTypeId = baseResourceTypeId;
            return this;
        }

        public Builder addSegmentId(long segmentId) {
            this.segmentIds.add(segmentId);
            return this;
        }

        public Builder addSegmentIds(Collection<? extends Long> segmentIds) {
            this.segmentIds.addAll(segmentIds);
            return this;
        }

        public Builder addSegmentIds(Long... segmentIds) {
            this.segmentIds.addAll(List.of(segmentIds));
            return this;
        }

        public Optional<Long> getId() {
            return Optional.ofNullable(id);
        }

        public Optional<String> getKey() {
            return Optional.ofNullable(key);
        }

        public Optional<String> getName() {
            return Optional.ofNullable(name);
        }

        public Optional<Long> getBaseResourceTypeId() {
            return Optional.ofNullable(baseResourceTypeId);
        }

        public Set<Long> getSegmentIds() {
            return segmentIds;
        }

        public BaseResource build() {
            Objects.requireNonNull(id, "Id is required");
            Objects.requireNonNull(key, "Key is required");
            Objects.requireNonNull(name, "Name is required");
            Objects.requireNonNull(baseResourceTypeId, "Base resource type id is required");
            return new BaseResource(id, key, name, baseResourceTypeId, segmentIds);
        }

        public BaseResource build(long id) {
            Objects.requireNonNull(key, "Key is required");
            Objects.requireNonNull(name, "Name is required");
            Objects.requireNonNull(baseResourceTypeId, "Base resource type id is required");
            return new BaseResource(id, key, name, baseResourceTypeId, segmentIds);
        }

    }

}
