package ru.yandex.qe.dispenser.domain.base_resources;

import java.util.Objects;
import java.util.Optional;

public final class BaseResourceChange {

    private final long id;
    private final long quotaRequestId;
    private final long bigOrderId;
    private final long baseResourceId;
    private final long amount;
    private final BaseResourceChangeByService amountByService;

    private BaseResourceChange(long id, long quotaRequestId, long bigOrderId, long baseResourceId, long amount,
                               BaseResourceChangeByService amountByService) {
        this.id = id;
        this.quotaRequestId = quotaRequestId;
        this.bigOrderId = bigOrderId;
        this.baseResourceId = baseResourceId;
        this.amount = amount;
        this.amountByService = amountByService;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(BaseResourceChange value) {
        return new Builder(value);
    }

    public static Update update(BaseResourceChange original) {
        return new Update(original);
    }

    public static Update update(BaseResourceChange original, BaseResourceChange updated) {
        return new Update(original)
                .amount(updated.amount)
                .amountByService(updated.amountByService);
    }

    public long getId() {
        return id;
    }

    public long getQuotaRequestId() {
        return quotaRequestId;
    }

    public long getBigOrderId() {
        return bigOrderId;
    }

    public long getBaseResourceId() {
        return baseResourceId;
    }

    public long getAmount() {
        return amount;
    }

    public BaseResourceChangeByService getAmountByService() {
        return amountByService;
    }

    public Key toKey() {
        return new Key(quotaRequestId, bigOrderId, baseResourceId);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        BaseResourceChange that = (BaseResourceChange) o;
        return id == that.id &&
                quotaRequestId == that.quotaRequestId &&
                bigOrderId == that.bigOrderId &&
                baseResourceId == that.baseResourceId &&
                amount == that.amount &&
                Objects.equals(amountByService, that.amountByService);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, quotaRequestId, bigOrderId, baseResourceId, amount, amountByService);
    }

    @Override
    public String toString() {
        return "BaseResourceChange{" +
                "id=" + id +
                ", quotaRequestId=" + quotaRequestId +
                ", bigOrderId=" + bigOrderId +
                ", baseResourceId=" + baseResourceId +
                ", amount=" + amount +
                ", amountByService=" + amountByService +
                '}';
    }

    public static final class Builder {

        private Long id;
        private Long quotaRequestId;
        private Long bigOrderId;
        private Long baseResourceId;
        private Long amount;
        private BaseResourceChangeByService amountByService;

        private Builder() {
        }

        private Builder(BaseResourceChange value) {
            this.id = value.id;
            this.quotaRequestId = value.quotaRequestId;
            this.bigOrderId = value.bigOrderId;
            this.baseResourceId = value.baseResourceId;
            this.amount = value.amount;
            this.amountByService = value.amountByService;
        }

        public Builder id(long id) {
            this.id = id;
            return this;
        }

        public Builder quotaRequestId(long quotaRequestId) {
            this.quotaRequestId = quotaRequestId;
            return this;
        }

        public Builder bigOrderId(long bigOrderId) {
            this.bigOrderId = bigOrderId;
            return this;
        }

        public Builder baseResourceId(long baseResourceId) {
            this.baseResourceId = baseResourceId;
            return this;
        }

        public Builder amount(long amount) {
            this.amount = amount;
            return this;
        }

        public Builder amountByService(BaseResourceChangeByService amountByService) {
            this.amountByService = amountByService;
            return this;
        }

        public Optional<Long> getId() {
            return Optional.ofNullable(id);
        }

        public Optional<Long> getQuotaRequestId() {
            return Optional.ofNullable(quotaRequestId);
        }

        public Optional<Long> getBigOrderId() {
            return Optional.ofNullable(bigOrderId);
        }

        public Optional<Long> getBaseResourceId() {
            return Optional.ofNullable(baseResourceId);
        }

        public Optional<Long> getAmount() {
            return Optional.ofNullable(amount);
        }

        public Optional<BaseResourceChangeByService> getAmountByService() {
            return Optional.ofNullable(amountByService);
        }

        public Key toKey() {
            Objects.requireNonNull(quotaRequestId, "Quota request id is required");
            Objects.requireNonNull(bigOrderId, "Big order id is required");
            Objects.requireNonNull(baseResourceId, "Base resource id is required");
            return new Key(quotaRequestId, bigOrderId, baseResourceId);
        }

        public BaseResourceChange build() {
            Objects.requireNonNull(id, "Id is required");
            Objects.requireNonNull(quotaRequestId, "Quota request id is required");
            Objects.requireNonNull(bigOrderId, "Big order id is required");
            Objects.requireNonNull(baseResourceId, "Base resource id is required");
            Objects.requireNonNull(amount, "Amount is required");
            Objects.requireNonNull(amountByService, "Amount by service is required");
            return new BaseResourceChange(id, quotaRequestId, bigOrderId, baseResourceId, amount, amountByService);
        }

        public BaseResourceChange build(long id) {
            Objects.requireNonNull(quotaRequestId, "Quota request id is required");
            Objects.requireNonNull(bigOrderId, "Big order id is required");
            Objects.requireNonNull(baseResourceId, "Base resource id is required");
            Objects.requireNonNull(amount, "Amount is required");
            Objects.requireNonNull(amountByService, "Amount by service is required");
            return new BaseResourceChange(id, quotaRequestId, bigOrderId, baseResourceId, amount, amountByService);
        }

    }

    public static final class Update {

        private final BaseResourceChange.Builder builder;

        private Update(BaseResourceChange original) {
            builder = BaseResourceChange.builder(original);
        }

        public Update amount(long amount) {
            builder.amount(amount);
            return this;
        }

        public Update amountByService(BaseResourceChangeByService amountByService) {
            builder.amountByService(amountByService);
            return this;
        }

        public BaseResourceChange build() {
            return builder.build();
        }

    }

    public static final class Key {

        private final long quotaRequestId;
        private final long bigOrderId;
        private final long baseResourceId;

        public Key(long quotaRequestId, long bigOrderId, long baseResourceId) {
            this.quotaRequestId = quotaRequestId;
            this.bigOrderId = bigOrderId;
            this.baseResourceId = baseResourceId;
        }

        public long getQuotaRequestId() {
            return quotaRequestId;
        }

        public long getBigOrderId() {
            return bigOrderId;
        }

        public long getBaseResourceId() {
            return baseResourceId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Key key = (Key) o;
            return quotaRequestId == key.quotaRequestId &&
                    bigOrderId == key.bigOrderId &&
                    baseResourceId == key.baseResourceId;
        }

        @Override
        public int hashCode() {
            return Objects.hash(quotaRequestId, bigOrderId, baseResourceId);
        }

        @Override
        public String toString() {
            return "Key{" +
                    "quotaRequestId=" + quotaRequestId +
                    ", bigOrderId=" + bigOrderId +
                    ", baseResourceId=" + baseResourceId +
                    '}';
        }

    }

}
