package ru.yandex.qe.dispenser.domain.base_resources;

import java.util.Objects;
import java.util.Optional;

public final class BaseResourceLimit {

    private final long id;
    private final long baseResourceId;
    private final long campaignId;
    private final long limit;

    private BaseResourceLimit(long id, long baseResourceId, long campaignId, long limit) {
        this.id = id;
        this.baseResourceId = baseResourceId;
        this.campaignId = campaignId;
        this.limit = limit;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(BaseResourceLimit value) {
        return new Builder(value);
    }

    public static Update update(BaseResourceLimit original) {
        return new Update(original);
    }

    public long getId() {
        return id;
    }

    public long getBaseResourceId() {
        return baseResourceId;
    }

    public long getCampaignId() {
        return campaignId;
    }

    public long getLimit() {
        return limit;
    }

    public Key toKey() {
        return new Key(baseResourceId, campaignId);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        BaseResourceLimit that = (BaseResourceLimit) o;
        return id == that.id &&
                baseResourceId == that.baseResourceId &&
                campaignId == that.campaignId &&
                limit == that.limit;
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, baseResourceId, campaignId, limit);
    }

    @Override
    public String toString() {
        return "BaseResourceLimit{" +
                "id=" + id +
                ", baseResourceId=" + baseResourceId +
                ", campaignId=" + campaignId +
                ", limit=" + limit +
                '}';
    }

    public static final class Builder {

        private Long id;
        private Long baseResourceId;
        private Long campaignId;
        private Long limit;

        private Builder() {
        }

        private Builder(BaseResourceLimit value) {
            this.id = value.id;
            this.baseResourceId = value.baseResourceId;
            this.campaignId = value.campaignId;
            this.limit = value.limit;
        }

        public Builder id(long id) {
            this.id = id;
            return this;
        }

        public Builder baseResourceId(long baseResourceId) {
            this.baseResourceId = baseResourceId;
            return this;
        }

        public Builder campaignId(long campaignId) {
            this.campaignId = campaignId;
            return this;
        }

        public Builder limit(long limit) {
            this.limit = limit;
            return this;
        }

        public Optional<Long> getId() {
            return Optional.ofNullable(id);
        }

        public Optional<Long> getBaseResourceId() {
            return Optional.ofNullable(baseResourceId);
        }

        public Optional<Long> getCampaignId() {
            return Optional.ofNullable(campaignId);
        }

        public Optional<Long> getLimit() {
            return Optional.ofNullable(limit);
        }

        public BaseResourceLimit build() {
            Objects.requireNonNull(id, "Id is required");
            Objects.requireNonNull(baseResourceId, "Base resource id is required");
            Objects.requireNonNull(campaignId, "Campaign id is required");
            Objects.requireNonNull(limit, "Limit is required");
            return new BaseResourceLimit(id, baseResourceId, campaignId, limit);
        }

        public BaseResourceLimit build(long id) {
            Objects.requireNonNull(baseResourceId, "Base resource id is required");
            Objects.requireNonNull(campaignId, "Campaign id is required");
            Objects.requireNonNull(limit, "Limit is required");
            return new BaseResourceLimit(id, baseResourceId, campaignId, limit);
        }

    }

    public static final class Update {

        private final BaseResourceLimit.Builder builder;

        private Update(BaseResourceLimit original) {
            this.builder = BaseResourceLimit.builder(original);
        }

        public Update limit(long limit) {
            builder.limit(limit);
            return this;
        }

        public BaseResourceLimit build() {
            return builder.build();
        }
    }

    public static final class Key {

        private final long baseResourceId;
        private final long campaignId;

        public Key(long baseResourceId, long campaignId) {
            this.baseResourceId = baseResourceId;
            this.campaignId = campaignId;
        }

        public long getBaseResourceId() {
            return baseResourceId;
        }

        public long getCampaignId() {
            return campaignId;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Key key = (Key) o;
            return baseResourceId == key.baseResourceId &&
                    campaignId == key.campaignId;
        }

        @Override
        public int hashCode() {
            return Objects.hash(baseResourceId, campaignId);
        }

        @Override
        public String toString() {
            return "Key{" +
                    "baseResourceId=" + baseResourceId +
                    ", campaignId=" + campaignId +
                    '}';
        }

    }

}
