package ru.yandex.qe.dispenser.domain.base_resources;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;

@JsonIgnoreProperties(ignoreUnknown = true)
public final class BaseResourceLinearRelationTerm {

    private final long numerator;
    private final long denominator;
    private final long resourceId;
    private final Set<Long> segmentIds;

    @JsonCreator
    public BaseResourceLinearRelationTerm(@JsonProperty("numerator") long numerator,
                                          @JsonProperty("denominator") long denominator,
                                          @JsonProperty("resourceId") long resourceId,
                                          @JsonProperty("segmentIds") Set<Long> segmentIds) {
        this.numerator = numerator;
        this.denominator = denominator;
        this.resourceId = resourceId;
        this.segmentIds = segmentIds;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(BaseResourceLinearRelationTerm value) {
        return new Builder(value);
    }

    public long getNumerator() {
        return numerator;
    }

    public long getDenominator() {
        return denominator;
    }

    public long getResourceId() {
        return resourceId;
    }

    public Set<Long> getSegmentIds() {
        return segmentIds;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        BaseResourceLinearRelationTerm that = (BaseResourceLinearRelationTerm) o;
        return numerator == that.numerator &&
                denominator == that.denominator &&
                resourceId == that.resourceId &&
                Objects.equals(segmentIds, that.segmentIds);
    }

    @Override
    public int hashCode() {
        return Objects.hash(numerator, denominator, resourceId, segmentIds);
    }

    @Override
    public String toString() {
        return "BaseResourceLinearRelationTerm{" +
                "numerator=" + numerator +
                ", denominator=" + denominator +
                ", resourceId=" + resourceId +
                ", segmentIds=" + segmentIds +
                '}';
    }

    public static final class Builder {

        private final Set<Long> segmentIds = new HashSet<>();

        private Long numerator;
        private Long denominator;
        private Long resourceId;

        private Builder() {
        }

        private Builder(BaseResourceLinearRelationTerm value) {
            this.numerator = value.numerator;
            this.denominator = value.denominator;
            this.resourceId = value.resourceId;
            this.segmentIds.addAll(value.segmentIds != null ? value.segmentIds : Set.of());
        }

        public Builder numerator(long numerator) {
            this.numerator = numerator;
            return this;
        }

        public Builder denominator(long denominator) {
            this.denominator = denominator;
            return this;
        }

        public Builder resourceId(long resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public Builder addSegmentId(long segmentId) {
            this.segmentIds.add(segmentId);
            return this;
        }

        public Builder addSegmentIds(Collection<? extends Long> segmentIds) {
            this.segmentIds.addAll(segmentIds);
            return this;
        }

        public Builder addSegmentIds(Long... segmentIds) {
            this.segmentIds.addAll(List.of(segmentIds));
            return this;
        }

        public BaseResourceLinearRelationTerm build() {
            Objects.requireNonNull(numerator, "Numerator is required");
            Objects.requireNonNull(denominator, "Denominator is required");
            Objects.requireNonNull(resourceId, "Resource id is required");
            return new BaseResourceLinearRelationTerm(numerator, denominator, resourceId, segmentIds);
        }

    }

}
