package ru.yandex.qe.dispenser.domain.base_resources;

import java.util.Objects;
import java.util.Optional;

public final class BaseResourceMapping {

    private final long id;
    private final long baseResourceId;
    private final long campaignId;
    private final long bigOrderId;
    private final long serviceId;
    private final BaseResourceRelation relation;

    private BaseResourceMapping(long id, long baseResourceId, long campaignId, long bigOrderId, long serviceId,
                                BaseResourceRelation relation) {
        this.id = id;
        this.baseResourceId = baseResourceId;
        this.campaignId = campaignId;
        this.bigOrderId = bigOrderId;
        this.serviceId = serviceId;
        this.relation = relation;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(BaseResourceMapping value) {
        return new Builder(value);
    }

    public long getId() {
        return id;
    }

    public long getBaseResourceId() {
        return baseResourceId;
    }

    public long getCampaignId() {
        return campaignId;
    }

    public long getBigOrderId() {
        return bigOrderId;
    }

    public long getServiceId() {
        return serviceId;
    }

    public BaseResourceRelation getRelation() {
        return relation;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        BaseResourceMapping that = (BaseResourceMapping) o;
        return id == that.id &&
                baseResourceId == that.baseResourceId &&
                campaignId == that.campaignId &&
                bigOrderId == that.bigOrderId &&
                serviceId == that.serviceId &&
                Objects.equals(relation, that.relation);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, baseResourceId, campaignId, bigOrderId, serviceId, relation);
    }

    @Override
    public String toString() {
        return "BaseResourceMapping{" +
                "id=" + id +
                ", baseResourceId=" + baseResourceId +
                ", campaignId=" + campaignId +
                ", bigOrderId=" + bigOrderId +
                ", serviceId=" + serviceId +
                ", relation=" + relation +
                '}';
    }

    public static final class Builder {

        private Long id;
        private Long baseResourceId;
        private Long campaignId;
        private Long bigOrderId;
        private Long serviceId;
        private BaseResourceRelation relation;

        private Builder() {
        }

        private Builder(BaseResourceMapping value) {
            this.id = value.id;
            this.baseResourceId = value.baseResourceId;
            this.campaignId = value.campaignId;
            this.bigOrderId = value.bigOrderId;
            this.serviceId = value.serviceId;
            this.relation = value.relation;
        }

        public Builder id(long id) {
            this.id = id;
            return this;
        }

        public Builder baseResourceId(long baseResourceId) {
            this.baseResourceId = baseResourceId;
            return this;
        }

        public Builder campaignId(long campaignId) {
            this.campaignId = campaignId;
            return this;
        }

        public Builder bigOrderId(long bigOrderId) {
            this.bigOrderId = bigOrderId;
            return this;
        }

        public Builder serviceId(long serviceId) {
            this.serviceId = serviceId;
            return this;
        }

        public Builder relation(BaseResourceRelation relation) {
            this.relation = relation;
            return this;
        }

        public Optional<Long> getId() {
            return Optional.ofNullable(id);
        }

        public Optional<Long> getBaseResourceId() {
            return Optional.ofNullable(baseResourceId);
        }

        public Optional<Long> getCampaignId() {
            return Optional.ofNullable(campaignId);
        }

        public Optional<Long> getBigOrderId() {
            return Optional.ofNullable(bigOrderId);
        }

        public Optional<Long> getServiceId() {
            return Optional.ofNullable(serviceId);
        }

        public Optional<BaseResourceRelation> getRelation() {
            return Optional.ofNullable(relation);
        }

        public BaseResourceMapping build() {
            Objects.requireNonNull(id, "Id is required");
            Objects.requireNonNull(baseResourceId, "Base resource id is required");
            Objects.requireNonNull(campaignId, "Campaign id is required");
            Objects.requireNonNull(bigOrderId, "Big order id is required");
            Objects.requireNonNull(serviceId, "Service id is required");
            Objects.requireNonNull(relation, "Relation is required");
            return new BaseResourceMapping(id, baseResourceId, campaignId, bigOrderId, serviceId, relation);
        }

        public BaseResourceMapping build(long id) {
            Objects.requireNonNull(baseResourceId, "Base resource id is required");
            Objects.requireNonNull(campaignId, "Campaign id is required");
            Objects.requireNonNull(bigOrderId, "Big order id is required");
            Objects.requireNonNull(serviceId, "Service id is required");
            Objects.requireNonNull(relation, "Relation is required");
            return new BaseResourceMapping(id, baseResourceId, campaignId, bigOrderId, serviceId, relation);
        }

    }

}
