package ru.yandex.qe.dispenser.domain.base_resources;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;

@JsonIgnoreProperties(ignoreUnknown = true)
public final class BaseResourceMdsRelationTerm {

    private final long resourceId;
    private final Set<Long> segmentIds;
    private final MdsLocation location;
    private final MdsStorageType storageType;
    private final long numerator;
    private final long denominator;
    private final Long abcServiceId;

    @JsonCreator
    public BaseResourceMdsRelationTerm(@JsonProperty("resourceId") long resourceId,
                                       @JsonProperty("segmentIds") Set<Long> segmentIds,
                                       @JsonProperty("location") MdsLocation location,
                                       @JsonProperty("storageType") MdsStorageType storageType,
                                       @JsonProperty("numerator") long numerator,
                                       @JsonProperty("denominator") long denominator,
                                       @JsonProperty("abcServiceId") Long abcServiceId) {
        this.resourceId = resourceId;
        this.segmentIds = segmentIds;
        this.location = location;
        this.storageType = storageType;
        this.numerator = numerator;
        this.denominator = denominator;
        this.abcServiceId = abcServiceId;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(BaseResourceMdsRelationTerm value) {
        return new Builder(value);
    }

    public long getResourceId() {
        return resourceId;
    }

    public Set<Long> getSegmentIds() {
        return segmentIds;
    }

    public MdsLocation getLocation() {
        return location;
    }

    public MdsStorageType getStorageType() {
        return storageType;
    }

    public long getNumerator() {
        return numerator;
    }

    public long getDenominator() {
        return denominator;
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Optional<Long> getAbcServiceId() {
        return Optional.ofNullable(abcServiceId);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        BaseResourceMdsRelationTerm that = (BaseResourceMdsRelationTerm) o;
        return resourceId == that.resourceId &&
                numerator == that.numerator &&
                denominator == that.denominator &&
                Objects.equals(segmentIds, that.segmentIds) &&
                location == that.location &&
                storageType == that.storageType &&
                Objects.equals(abcServiceId, that.abcServiceId);
    }

    @Override
    public int hashCode() {
        return Objects.hash(resourceId, segmentIds, location, storageType, numerator, denominator, abcServiceId);
    }

    @Override
    public String toString() {
        return "BaseResourceMdsRelationTerm{" +
                "resourceId=" + resourceId +
                ", segmentIds=" + segmentIds +
                ", location=" + location +
                ", storageType=" + storageType +
                ", numerator=" + numerator +
                ", denominator=" + denominator +
                ", abcServiceId=" + abcServiceId +
                '}';
    }

    public static final class Builder {

        private final Set<Long> segmentIds = new HashSet<>();

        private Long resourceId;
        private MdsLocation location;
        private MdsStorageType storageType;
        private Long numerator;
        private Long denominator;
        private Long abcServiceId;

        private Builder() {
        }

        private Builder(BaseResourceMdsRelationTerm value) {
            this.resourceId = value.resourceId;
            this.segmentIds.addAll(value.segmentIds != null ? value.segmentIds : Set.of());
        }

        public Builder resourceId(long resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public Builder location(MdsLocation location) {
            this.location = location;
            return this;
        }

        public Builder storageType(MdsStorageType storageType) {
            this.storageType = storageType;
            return this;
        }

        public Builder numerator(long numerator) {
            this.numerator = numerator;
            return this;
        }

        public Builder denominator(long denominator) {
            this.denominator = denominator;
            return this;
        }

        public Builder abcServiceId(Long abcServiceId) {
            this.abcServiceId = abcServiceId;
            return this;
        }

        public Builder addSegmentId(long segmentId) {
            this.segmentIds.add(segmentId);
            return this;
        }

        public Builder addSegmentIds(Collection<? extends Long> segmentIds) {
            this.segmentIds.addAll(segmentIds);
            return this;
        }

        public Builder addSegmentIds(Long... segmentIds) {
            this.segmentIds.addAll(List.of(segmentIds));
            return this;
        }

        public BaseResourceMdsRelationTerm build() {
            Objects.requireNonNull(resourceId, "Resource id is required");
            Objects.requireNonNull(location, "Location is required");
            Objects.requireNonNull(storageType, "Storage type is required");
            Objects.requireNonNull(numerator, "Numerator is required");
            Objects.requireNonNull(denominator, "Denominator is required");
            return new BaseResourceMdsRelationTerm(resourceId, segmentIds, location, storageType,
                    numerator, denominator, abcServiceId);
        }

    }

}
