package ru.yandex.qe.dispenser.domain.base_resources;

import java.math.BigDecimal;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Preconditions;

@JsonIgnoreProperties(ignoreUnknown = true)
public final class BaseResourceRelation {

    private final BaseResourceLinearRelation linear;
    private final BaseResourceMdsRelation mds;

    @JsonCreator
    public BaseResourceRelation(@JsonProperty("linear") BaseResourceLinearRelation linear,
                                @JsonProperty("mds") BaseResourceMdsRelation mds) {
        this.linear = linear;
        this.mds = mds;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(BaseResourceRelation value) {
        return new Builder(value);
    }

    public Optional<BaseResourceLinearRelation> getLinear() {
        return Optional.ofNullable(linear);
    }

    public Optional<BaseResourceMdsRelation> getMds() {
        return Optional.ofNullable(mds);
    }

    @JsonIgnore
    public Set<ResourceAndSegmentsId> getAllResources() {
        if (linear != null) {
            return linear.getAllResources();
        } else if (mds != null) {
            return mds.getAllResources();
        }
        return Set.of();
    }

    @JsonIgnore
    public boolean isLocalEvaluation() {
        if (linear != null) {
            return linear.isLocalEvaluation();
        } else if (mds != null) {
            return mds.isLocalEvaluation();
        }
        return true;
    }

    public BigDecimal evaluateLocally(Map<ResourceAndSegmentsId, Long> parameters) {
        if (linear != null) {
            return linear.evaluateLocally(parameters);
        }
        return BigDecimal.ZERO;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        BaseResourceRelation that = (BaseResourceRelation) o;
        return Objects.equals(linear, that.linear) &&
                Objects.equals(mds, that.mds);
    }

    @Override
    public int hashCode() {
        return Objects.hash(linear, mds);
    }

    @Override
    public String toString() {
        return "BaseResourceRelation{" +
                "linear=" + linear +
                ", mds=" + mds +
                '}';
    }

    public static final class Builder {

        private BaseResourceLinearRelation linear;
        private BaseResourceMdsRelation mds;

        private Builder() {
        }

        private Builder(BaseResourceRelation value) {
            this.linear = value.linear;
            this.mds = value.mds;
        }

        public Builder linear(BaseResourceLinearRelation linear) {
            this.linear = linear;
            return this;
        }

        public Builder mds(BaseResourceMdsRelation mds) {
            this.mds = mds;
            return this;
        }

        public BaseResourceRelation build() {
            Preconditions.checkArgument(linear != null ^ mds != null, "Exactly one type of relation is required");
            return new BaseResourceRelation(linear, mds);
        }

    }

}
