package ru.yandex.qe.dispenser.domain.base_resources;

import java.util.Objects;
import java.util.Optional;

import ru.yandex.qe.dispenser.api.v1.DiResourceType;

public final class BaseResourceType {

    private final long id;
    private final String key;
    private final String name;
    private final long serviceId;
    private final DiResourceType resourceType;

    private BaseResourceType(long id, String key, String name, long serviceId, DiResourceType resourceType) {
        this.id = id;
        this.key = key;
        this.name = name;
        this.serviceId = serviceId;
        this.resourceType = resourceType;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(BaseResourceType value) {
        return new Builder(value);
    }

    public long getId() {
        return id;
    }

    public String getKey() {
        return key;
    }

    public String getName() {
        return name;
    }

    public long getServiceId() {
        return serviceId;
    }

    public DiResourceType getResourceType() {
        return resourceType;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        BaseResourceType that = (BaseResourceType) o;
        return id == that.id &&
                serviceId == that.serviceId &&
                Objects.equals(key, that.key) &&
                Objects.equals(name, that.name) &&
                resourceType == that.resourceType;
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, key, name, serviceId, resourceType);
    }

    @Override
    public String toString() {
        return "BaseResourceType{" +
                "id=" + id +
                ", key='" + key + '\'' +
                ", name='" + name + '\'' +
                ", serviceId=" + serviceId +
                ", resourceType=" + resourceType +
                '}';
    }

    public static final class Builder {

        private Long id;
        private String key;
        private String name;
        private Long serviceId;
        private DiResourceType resourceType;

        private Builder() {
        }

        private Builder(BaseResourceType value) {
            this.id = value.id;
            this.key = value.key;
            this.name = value.name;
            this.serviceId = value.serviceId;
            this.resourceType = value.resourceType;
        }

        public Builder id(long id) {
            this.id = id;
            return this;
        }

        public Builder key(String key) {
            this.key = key;
            return this;
        }

        public Builder name(String name) {
            this.name = name;
            return this;
        }

        public Builder serviceId(long serviceId) {
            this.serviceId = serviceId;
            return this;
        }

        public Builder resourceType(DiResourceType resourceType) {
            this.resourceType = resourceType;
            return this;
        }

        public Optional<Long> getId() {
            return Optional.ofNullable(id);
        }

        public Optional<String> getKey() {
            return Optional.ofNullable(key);
        }

        public Optional<String> getName() {
            return Optional.ofNullable(name);
        }

        public Optional<Long> getServiceId() {
            return Optional.ofNullable(serviceId);
        }

        public Optional<DiResourceType> getResourceType() {
            return Optional.ofNullable(resourceType);
        }

        public BaseResourceType build() {
            Objects.requireNonNull(id, "Id is required");
            Objects.requireNonNull(key, "Key is required");
            Objects.requireNonNull(name, "Name is required");
            Objects.requireNonNull(serviceId, "Service id is required");
            Objects.requireNonNull(resourceType, "Resource type is required");
            return new BaseResourceType(id, key, name, serviceId, resourceType);
        }

        public BaseResourceType build(long id) {
            Objects.requireNonNull(key, "Key is required");
            Objects.requireNonNull(name, "Name is required");
            Objects.requireNonNull(serviceId, "Service id is required");
            Objects.requireNonNull(resourceType, "Resource type is required");
            return new BaseResourceType(id, key, name, serviceId, resourceType);
        }

    }

}
