package ru.yandex.qe.dispenser.domain.base_resources;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;

@JsonIgnoreProperties(ignoreUnknown = true)
public final class ServiceBaseResourceChange {

    private final long serviceId;
    private final long amount;
    private final Set<Long> mappingIds;

    @JsonCreator
    public ServiceBaseResourceChange(@JsonProperty("serviceId") long serviceId,
                                     @JsonProperty("amount") long amount,
                                     @JsonProperty("mappingIds") Set<Long> mappingIds) {
        this.serviceId = serviceId;
        this.amount = amount;
        this.mappingIds = mappingIds;
    }

    public static Builder builder() {
        return new Builder();
    }

    public static Builder builder(ServiceBaseResourceChange value) {
        return new Builder(value);
    }

    public long getServiceId() {
        return serviceId;
    }

    public long getAmount() {
        return amount;
    }

    public Set<Long> getMappingIds() {
        return mappingIds;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        ServiceBaseResourceChange that = (ServiceBaseResourceChange) o;
        return serviceId == that.serviceId &&
                amount == that.amount &&
                Objects.equals(mappingIds, that.mappingIds);
    }

    @Override
    public int hashCode() {
        return Objects.hash(serviceId, amount, mappingIds);
    }

    @Override
    public String toString() {
        return "ServiceBaseResourceChange{" +
                "serviceId=" + serviceId +
                ", amount=" + amount +
                ", mappingIds=" + mappingIds +
                '}';
    }

    public static final class Builder {

        private final Set<Long> mappingIds = new HashSet<>();

        private Long serviceId;
        private Long amount;

        private Builder() {
        }

        private Builder(ServiceBaseResourceChange value) {
            this.serviceId = value.serviceId;
            this.amount = value.amount;
            this.mappingIds.addAll(value.mappingIds != null ? value.mappingIds : Set.of());
        }

        public Builder serviceId(long serviceId) {
            this.serviceId = serviceId;
            return this;
        }

        public Builder amount(long amount) {
            this.amount = amount;
            return this;
        }

        public Builder addMappingId(long mappingId) {
            this.mappingIds.add(mappingId);
            return this;
        }

        public Builder addMappingIds(Collection<? extends Long> mappingIds) {
            this.mappingIds.addAll(mappingIds);
            return this;
        }

        public Builder addMappingIds(Long... mappingIds) {
            this.mappingIds.addAll(List.of(mappingIds));
            return this;
        }

        public ServiceBaseResourceChange build() {
            Objects.requireNonNull(serviceId, "Service id is required");
            Objects.requireNonNull(amount, "Amount is required");
            return new ServiceBaseResourceChange(serviceId, amount, mappingIds);
        }

    }

}
