package ru.yandex.qe.dispenser.domain.bot;

import java.time.LocalDate;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.qe.dispenser.domain.index.LongIndexable;

@ParametersAreNonnullByDefault
public final class BigOrder implements LongIndexable {

    private final long id;
    private final LocalDate date;
    private final List<BigOrderConfig> configs;
    private final boolean deleted;

    private BigOrder(long id,
                     LocalDate date,
                     List<BigOrderConfig> configs,
                     boolean deleted) {
        this.id = id;
        this.date = date;
        this.configs = configs;
        this.deleted = deleted;
    }

    public static Builder builder(LocalDate date) {
        return new Builder(date);
    }

    public static Builder builder(BigOrder value) {
        return new Builder(value);
    }

    public static Update update(BigOrder current) {
        return new Update(current);
    }

    @Override
    public long getId() {
        return id;
    }

    public LocalDate getDate() {
        return date;
    }

    public List<BigOrderConfig> getConfigs() {
        return configs;
    }

    public boolean isDeleted() {
        return deleted;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        BigOrder that = (BigOrder) o;
        return id == that.id &&
                deleted == that.deleted &&
                Objects.equals(date, that.date) &&
                Objects.equals(configs, that.configs);
    }

    @Override
    public int hashCode() {
        return Objects.hash(id, date, configs, deleted);
    }

    @Override
    public String toString() {
        return "SimpleBigOrder{" +
                "id=" + id +
                ", date='" + date + '\'' +
                ", configs=" + configs +
                ", deleted=" + deleted +
                '}';
    }

    public static final class Builder {

        private LocalDate date;
        private List<BigOrderConfig> configs = new ArrayList<>();
        private boolean deleted = false;

        private Builder(LocalDate date) {
            this.date = date;
        }

        private Builder(BigOrder value) {
            this.date = value.date;
            this.configs = value.configs;
            this.deleted = value.deleted;
        }

        public Builder date(LocalDate date) {
            this.date = date;
            return this;
        }

        public Builder configs(List<BigOrderConfig> configs) {
            this.configs = configs;
            return this;
        }

        public Builder deleted(boolean deleted) {
            this.deleted = deleted;
            return this;
        }

        public LocalDate getDate() {
            return date;
        }

        public List<BigOrderConfig> getConfigs() {
            return configs;
        }

        public boolean isDeleted() {
            return deleted;
        }

        public BigOrder build(long id) {
            return new BigOrder(id, date, configs, deleted);
        }

    }

    public static final class Update {

        private final BigOrder.Builder builder;
        private final long id;

        private Update(BigOrder original) {
            this.builder = new Builder(original);
            this.id = original.id;
        }

        public Update date(LocalDate date) {
            this.builder.date(date);
            return this;
        }

        public Update configs(List<BigOrderConfig> configs) {
            this.builder.configs(configs);
            return this;
        }

        public Update deleted(boolean deleted) {
            this.builder.deleted(deleted);
            return this;
        }

        public BigOrder build() {
            return builder.build(id);
        }

    }

}
