package ru.yandex.qe.dispenser.domain.bot;

import java.io.Serializable;
import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.apache.commons.lang3.builder.CompareToBuilder;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import ru.yandex.qe.dispenser.api.v1.DiSegment;
import ru.yandex.qe.dispenser.api.v1.DiService;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Service;


/**
 * Link {@link Service} and Configuration on some {@link Segment}
 *
 * @see Service
 * @see Segment
 */
@ParametersAreNonnullByDefault
public class BotServiceConfiguration implements Serializable, Comparable<BotServiceConfiguration> {

    private final String key;
    private final Service service;
    private final Long configurationId;
    @Nullable
    private final Segment locationSegment;
    private final long botCampaignGroupId;
    private final boolean isValid;

    public BotServiceConfiguration(final String key,
                                   final Service service,
                                   final Long configurationId,
                                   @Nullable final Segment locationSegment,
                                   final long botCampaignGroupId,
                                   final boolean isValid) {
        this.key = key;
        this.service = service;
        this.configurationId = configurationId;
        this.locationSegment = locationSegment;
        this.botCampaignGroupId = botCampaignGroupId;
        this.isValid = isValid;
    }

    public Service getService() {
        return service;
    }

    public Long getConfigurationId() {
        return configurationId;
    }

    @Nullable
    public Segment getLocationSegment() {
        return locationSegment;
    }

    public boolean isValid() {
        return isValid;
    }

    public String getKey() {
        return key;
    }

    public long getBotCampaignGroupId() {
        return botCampaignGroupId;
    }

    @Override
    public int compareTo(@NotNull final BotServiceConfiguration o) {
        return new CompareToBuilder()
                .append(getBotCampaignGroupId(), o.getBotCampaignGroupId())
                .append(getService(), o.getService())
                .append(getConfigurationId(), o.getConfigurationId())
                .append(getLocationSegment(), o.getLocationSegment())
                .append(getKey(), o.getKey())
                .build();
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final BotServiceConfiguration that = (BotServiceConfiguration) o;
        return Objects.equals(service, that.service) &&
                Objects.equals(configurationId, that.configurationId) &&
                Objects.equals(locationSegment, that.locationSegment) &&
                Objects.equals(botCampaignGroupId, that.botCampaignGroupId) &&
                Objects.equals(key, that.key);
    }

    @Override
    public int hashCode() {
        return Objects.hash(service, configurationId, locationSegment, botCampaignGroupId, key);
    }

    public View toView(final ConfigurationWithComponents configurationWithComponents) {
        return new View(key, service.toView(), configurationWithComponents, locationSegment == null ? null : locationSegment.toView(), botCampaignGroupId);
    }

    public static class View {
        private final String key;
        private final DiService service;
        private final ConfigurationWithComponents configuration;
        @Nullable
        private final DiSegment locationSegment;
        private final long botCampaignGroupId;

        @JsonCreator
        public View(@JsonProperty("key") final String key,
                    @JsonProperty("service") final DiService service,
                    @JsonProperty("configuration") final ConfigurationWithComponents configuration,
                    @Nullable
                    @JsonProperty("locationSegment") final DiSegment locationSegment,
                    @JsonProperty("botCampaignGroupId") final long botCampaignGroupId) {
            this.key = key;
            this.service = service;
            this.configuration = configuration;
            this.locationSegment = locationSegment;
            this.botCampaignGroupId = botCampaignGroupId;
        }

        public DiService getService() {
            return service;
        }

        public ConfigurationWithComponents getConfiguration() {
            return configuration;
        }

        @Nullable
        public DiSegment getLocationSegment() {
            return locationSegment;
        }

        public String getKey() {
            return key;
        }

        public long getBotCampaignGroupId() {
            return botCampaignGroupId;
        }
    }

    public static class Body {
        private final String key;
        private final Long configurationId;
        @Nullable
        private final String locationSegmentKey;

        @JsonCreator
        public Body(@JsonProperty("key") final String key,
                    @JsonProperty("configurationId") final Long configurationId,
                    @Nullable @JsonProperty("locationSegmentKey") final String locationSegmentKey) {
            this.key = key;
            this.configurationId = configurationId;
            this.locationSegmentKey = locationSegmentKey;
        }

        public Long getConfigurationId() {
            return configurationId;
        }

        @Nullable
        public String getLocationSegmentKey() {
            return locationSegmentKey;
        }

        public String getKey() {
            return key;
        }
    }
}
