package ru.yandex.qe.dispenser.domain.bot;

import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

import ru.yandex.qe.dispenser.api.v1.DiBotSyncStatus;

public class BotSyncStatus {

    private final static DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ISO_DATE_TIME
            .withZone(ZoneId.of("Europe/Moscow"));

    private final DiBotSyncStatus.Status status;
    private final Instant date;
    private final List<String> errors;

    private BotSyncStatus(final Builder builder) {
        this.status = builder.status;
        this.date = builder.date;
        this.errors = builder.errors != null
                ? Collections.unmodifiableList(builder.errors)
                : Collections.emptyList();

        Objects.requireNonNull(status, "Status is required!");
        Objects.requireNonNull(date, "Date is required!");
    }

    public DiBotSyncStatus.Status getStatus() {
        return status;
    }

    public Instant getDate() {
        return date;
    }

    public List<String> getErrors() {
        return errors;
    }

    public static Builder builder() {
        return new Builder();
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final BotSyncStatus that = (BotSyncStatus) o;
        return status == that.status &&
                Objects.equals(date, that.date) &&
                Objects.equals(errors, that.errors);
    }

    public DiBotSyncStatus toView() {
        return DiBotSyncStatus.builder()
                .status(status)
                .date(DATE_TIME_FORMATTER.format(date))
                .errors(errors)
                .build();
    }

    @Override
    public int hashCode() {
        return Objects.hash(status, date, errors);
    }

    @Override
    public String toString() {
        return "BotSyncStatus{" +
                "status=" + status +
                ", date=" + date +
                ", errors=" + errors +
                '}';
    }

    public static class Builder {
        private DiBotSyncStatus.Status status;
        private Instant date;
        private List<String> errors;

        public Builder status(final DiBotSyncStatus.Status status) {
            this.status = status;
            return this;
        }

        public Builder date(final Instant date) {
            this.date = date;
            return this;
        }

        public Builder errors(final List<String> errors) {
            this.errors = errors;
            return this;
        }

        public Builder errors(final String error) {
            if (errors == null) {
                errors = new ArrayList<>();
            }

            errors.add(error);
            return this;
        }

        public BotSyncStatus build() {
            return new BotSyncStatus(this);
        }
    }
}
