package ru.yandex.qe.dispenser.domain.d;

import java.util.Objects;
import java.util.Optional;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.base.Preconditions;

@JsonIgnoreProperties(ignoreUnknown = true)
public final class DeliverableResponseDto {

    private final Long serviceId;
    private final String folderId;
    private final String resourceId;
    private final DeliverableFolderOperationDto folderOperationLog;
    private final DeliverableMetaResponseDto meta;

    @JsonCreator
    public DeliverableResponseDto(@JsonProperty("serviceId") Long serviceId,
                                  @JsonProperty("folderId") String folderId,
                                  @JsonProperty("resourceId") String resourceId,
                                  @JsonProperty("folderOperationLog") DeliverableFolderOperationDto folderOperationLog,
                                  @JsonProperty("meta") DeliverableMetaResponseDto meta) {
        this.serviceId = serviceId;
        this.folderId = folderId;
        this.resourceId = resourceId;
        this.folderOperationLog = folderOperationLog;
        this.meta = meta;
    }

    public static Builder builder() {
        return new Builder();
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Optional<Long> getServiceId() {
        return Optional.ofNullable(serviceId);
    }

    @JsonInclude(JsonInclude.Include.NON_EMPTY)
    public Optional<String> getFolderId() {
        return Optional.ofNullable(folderId);
    }

    public String getResourceId() {
        return resourceId;
    }

    public DeliverableFolderOperationDto getFolderOperationLog() {
        return folderOperationLog;
    }

    public DeliverableMetaResponseDto getMeta() {
        return meta;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        DeliverableResponseDto that = (DeliverableResponseDto) o;
        return Objects.equals(serviceId, that.serviceId) &&
                Objects.equals(folderId, that.folderId) &&
                Objects.equals(resourceId, that.resourceId) &&
                Objects.equals(folderOperationLog, that.folderOperationLog) &&
                Objects.equals(meta, that.meta);
    }

    @Override
    public int hashCode() {
        return Objects.hash(serviceId, folderId, resourceId, folderOperationLog, meta);
    }

    @Override
    public String toString() {
        return "DeliverableResponseDto{" +
                "serviceId=" + serviceId +
                ", folderId='" + folderId + '\'' +
                ", resourceId='" + resourceId + '\'' +
                ", folderOperationLog=" + folderOperationLog +
                ", meta=" + meta +
                '}';
    }

    public static final class Builder {

        private Long serviceId;
        private String folderId;
        private String resourceId;
        private DeliverableFolderOperationDto folderOperationLog;
        private DeliverableMetaResponseDto meta;

        private Builder() {
        }

        public Builder serviceId(long serviceId) {
            this.serviceId = serviceId;
            return this;
        }

        public Builder folderId(String folderId) {
            this.folderId = folderId;
            return this;
        }

        public Builder resourceId(String resourceId) {
            this.resourceId = resourceId;
            return this;
        }

        public Builder folderOperationLog(DeliverableFolderOperationDto folderOperationLog) {
            this.folderOperationLog = folderOperationLog;
            return this;
        }

        public Builder meta(DeliverableMetaResponseDto meta) {
            this.meta = meta;
            return this;
        }

        public DeliverableResponseDto build() {
            Objects.requireNonNull(resourceId, "Resource id is required");
            Objects.requireNonNull(folderOperationLog, "Folder operation log is required");
            Objects.requireNonNull(meta, "Meta is required");
            Preconditions.checkArgument(serviceId != null || folderId != null,
                    "Either service id or folder id is required");
            return new DeliverableResponseDto(serviceId, folderId, resourceId, folderOperationLog, meta);
        }

    }

}
