package ru.yandex.qe.dispenser.domain.dao;

import java.io.Serializable;
import java.util.Collection;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.jetbrains.annotations.NotNull;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.datasource.lookup.DataSourceLookupFailureException;

import ru.yandex.qe.dispenser.domain.util.CollectionUtils;

public class KeyExtractorInMemoryDaoImpl<T, K extends Comparable<K> & Serializable>
        implements InMemoryDao<T, K> {
    @NotNull
    protected final Map<K, T> id2obj = new ConcurrentHashMap<>();

    protected final Function<T, K> keyExtractor;

    public KeyExtractorInMemoryDaoImpl(final Function<T, K> keyExtractor) {
        this.keyExtractor = keyExtractor;
    }

    @NotNull
    @Override
    public T create(@NotNull final T newInstance) {
        final K key = keyExtractor.apply(newInstance);
        if (id2obj.containsKey(key)) {
            throw new DuplicateKeyException(
                    newInstance.getClass().getSimpleName() + " object with key " + key + " already exists!");
        }
        id2obj.put(key, newInstance);
        return newInstance;
    }

    @Override
    public boolean update(@NotNull final T trainsientObject) {
        final K key = keyExtractor.apply(trainsientObject);
        if (!id2obj.containsKey(key)) {
            throw new IllegalArgumentException(trainsientObject.getClass().getSimpleName() + " object not exists to update!");
        }
        id2obj.put(key, trainsientObject);
        return true;
    }

    @Override
    public boolean delete(@NotNull final T persistentObject) {
        final K key = keyExtractor.apply(persistentObject);
        if (id2obj.remove(key) == null) {
            throw new DataSourceLookupFailureException(
                    persistentObject.getClass().getSimpleName() + " with id '" + key + "' not exists to delete!");
        }
        return true;
    }

    @NotNull
    @Override
    public Set<T> getAll() {
        return CollectionUtils.copyToUnmodifiableSet(id2obj.values());
    }

    @NotNull
    @Override
    public T read(@NotNull final K key) throws EmptyResultDataAccessException {
        return filter(obj -> keyExtractor.apply(obj).equals(key))
                .findAny()
                .orElseThrow(() -> new EmptyResultDataAccessException(
                        "No object in " + getClass().getSimpleName() + " with key '" + key + "'!", 1));
    }

    @NotNull
    public Set<T> readAll(@NotNull final Collection<K> keys) {
        return filter(obj -> keys.contains(keyExtractor.apply(obj)))
                .collect(Collectors.toSet());
    }

    @Override
    public boolean clear() {
        id2obj.clear();
        return true;
    }
}
