package ru.yandex.qe.dispenser.domain.dao;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;

import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.ImmutableMap;
import org.apache.commons.io.IOUtils;
import org.apache.commons.text.StringEscapeUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.ResultSetExtractor;
import org.springframework.jdbc.core.RowMapper;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;

import ru.yandex.qe.dispenser.api.util.SerializationUtils;
import ru.yandex.qe.dispenser.api.v1.DiYandexGroupType;
import ru.yandex.qe.dispenser.domain.Person;
import ru.yandex.qe.dispenser.domain.PersonAffiliation;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Quota;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.YaGroup;
import ru.yandex.qe.dispenser.domain.index.LongIndexable;
import ru.yandex.qe.dispenser.domain.util.PageInfo;
import ru.yandex.qe.dispenser.domain.util.RelativePage;
import ru.yandex.qe.dispenser.domain.util.RelativePageInfo;

import static ru.yandex.qe.dispenser.domain.util.ValidationUtils.validateProjectKey;
import static ru.yandex.qe.dispenser.domain.util.ValidationUtils.validateResourceKey;

public abstract class SqlDaoBase {
    private static final String ACQUIRE_ROW_EXCLUSIVE_LOCK_QUERY_TEMPLATE = "LOCK TABLE %s IN ROW EXCLUSIVE MODE";

    protected static final String PAGE_SIZE_PARAM = "pageSize";
    protected static final String OFFSET_PARAM = "offset";

    @Autowired
    protected DiJdbcTemplate jdbcTemplate;

    public void lockForChanges() {
        throw new UnsupportedOperationException();
    }

    protected <T extends LongIndexable> RelativePage<T> getRelativePage(final String query, final Map<String, Object> params, final RowMapper<T> rowMapper,
                                                                        final RelativePageInfo pageInfo) {
        final Map<String, Object> paramsWithPageInfo = new HashMap<>(params);
        final long expectedPageSize = pageInfo.getPageSize() + 1;
        paramsWithPageInfo.put(PAGE_SIZE_PARAM, expectedPageSize);
        paramsWithPageInfo.put(OFFSET_PARAM, pageInfo.getFromId());
        final List<T> result = jdbcTemplate.query(query, paramsWithPageInfo, rowMapper);
        if (result.size() < expectedPageSize) {
            return RelativePage.last(result);
        }
        return RelativePage.middle(result.subList(0, (int) pageInfo.getPageSize()));
    }

    protected <T extends LongIndexable, C extends List<T>> RelativePage<T> getRelativePage(final String query,
                                                                                               final Map<String, Object> params,
                                                                                               final ResultSetExtractor<C> rse,
                                                                                               final RelativePageInfo pageInfo) {
        final Map<String, Object> paramsWithPageInfo = new HashMap<>(params);
        final long expectedPageSize = pageInfo.getPageSize() + 1;
        paramsWithPageInfo.put(PAGE_SIZE_PARAM, expectedPageSize);
        paramsWithPageInfo.put(OFFSET_PARAM, pageInfo.getFromId());
        final C result = jdbcTemplate.query(query, new MapSqlParameterSource(paramsWithPageInfo), rse);
        if (result.size() < expectedPageSize) {
            return RelativePage.last(result);
        }
        return RelativePage.middle(result.subList(0, (int) pageInfo.getPageSize()));
    }

    protected long toId(@NotNull final ResultSet rs, final int i) throws SQLException {
        return rs.getLong("id");
    }

    @NotNull
    protected Person toPerson(@NotNull final ResultSet rs) throws SQLException {
        return new Person(rs.getLong("id"), rs.getString("login"), rs.getLong("uid"),
                rs.getBoolean("is_robot"), rs.getBoolean("is_dismissed"), rs.getBoolean("is_deleted"),
                PersonAffiliation.valueOf(rs.getString("affiliation")));
    }

    @NotNull
    protected Person toPerson(@NotNull final ResultSet rs, final int i) throws SQLException {
        return toPerson(rs);
    }

    @NotNull
    protected YaGroup toGroup(@NotNull final ResultSet rs) throws SQLException {
        final YaGroup group = new YaGroup(rs.getString("url_acceptable_key"), DiYandexGroupType.valueOf(rs.getString("group_type")),
                rs.getLong("staff_id"), rs.getBoolean("deleted"));
        group.setId(rs.getLong("id"));
        return group;
    }

    @NotNull
    protected YaGroup toGroup(@NotNull final ResultSet rs, final int i) throws SQLException {
        return toGroup(rs);
    }

    @NotNull
    protected Map<String, Object> toParams(final @NotNull Project project) {
        validateProjectKey(project.getPublicKey());
        return new HashMap<String, Object>() {{
            put("projectId", project.getId());
            put("key", project.getPublicKey());
            put("shortName", project.getName());
            put("description", project.getDescription());
            put("parentId", !project.isRoot() ? project.getParent().getId() : null);
            put("personId", project.isPersonal() ? project.getPerson().getId() : null);
            put("abcServiceId", project.getAbcServiceId());
            put("removed", project.isRemoved());
            put("syncedWithAbc", project.isSyncedWithAbc());
            put("mailList", project.getMailList());
            put("valueStreamAbcServiceId", project.getValueStreamAbcServiceId());
        }};
    }

    @NotNull
    protected Map<String, Object> toParams(final @NotNull Resource.Key key) {
        validateResourceKey(key.getPublicKey());
        return ImmutableMap.<String, Object>builder()
                .put("key", key.getPublicKey())
                .put("serviceId", key.getService().getId())
                .build();
    }

    @NotNull
    protected Map<String, Object> toParams(final @NotNull Resource resource) {
        final Map<String, Object> params = new HashMap<>();
        params.put("resourceId", resource.getId());
        params.put("name", resource.getName());
        params.put("description", resource.getDescription());
        params.put("resourceType", resource.getType().name());
        params.put("quotingMode", resource.getMode().name());
        params.put("resourceGroupId", resource.getGroup() == null ? null : resource.getGroup().getId());
        params.put("priority", resource.getPriority());
        params.putAll(toParams(resource.getKey()));
        return params;
    }

    @NotNull
    protected Map<String, Object> toParams(final @NotNull Quota quota) {
        return new HashMap<String, Object>() {{
            put("id", quota.getId());
            put("quotaSpecId", quota.getSpec().getId());
            put("ownActualValue", quota.getOwnActual());
            put("maxValue", quota.getMax());
            put("ownMaxValue", quota.getOwnMax());
            put("lastOverquotingTs", SqlUtils.toTimestamp(quota.getLastOverquotingTs()));
            putAll(toParams(quota.getProject()));
        }};
    }

    protected static Map<String, Object> toParams(final PageInfo pageInfo) {
        final Map<String, Object> params = new HashMap<>();
        params.put(OFFSET_PARAM, pageInfo.getOffset());
        params.put(PAGE_SIZE_PARAM, pageInfo.getPageSize());
        return params;
    }

    @NotNull
    public <T> SqlParameterSource[] toBatchParams(@NotNull final Collection<T> col, final @NotNull Function<T, Map<String, ?>> toParams) {
        return col.stream().map(toParams).map(MapSqlParameterSource::new).toArray(SqlParameterSource[]::new);
    }

    @NotNull
    protected static String query(@NotNull final String classpath, @NotNull final Object... args) {
        try {
            return String.format(IOUtils.toString(SqlDaoBase.class.getResourceAsStream(classpath), StandardCharsets.UTF_8), args);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    protected void acquireRowExclusiveLockOnTable(@NotNull final String tableName) {
        jdbcTemplate.execute(String.format(ACQUIRE_ROW_EXCLUSIVE_LOCK_QUERY_TEMPLATE, tableName));
    }

    @Nullable
    public static Integer getInteger(@NotNull final ResultSet rs, @NotNull final String label) throws SQLException {
        final int value = rs.getInt(label);
        if (rs.wasNull()) {
            return null;
        }
        return value;
    }

    @Nullable
    public static Short getShort(@NotNull final ResultSet rs, @NotNull final String label) throws SQLException {
        final short value = rs.getShort(label);
        if (rs.wasNull()) {
            return null;
        }
        return value;
    }

    @Nullable
    public static Long getLong(@NotNull final ResultSet rs, @NotNull final String label) throws SQLException {
        final long value = rs.getLong(label);
        if (rs.wasNull()) {
            return null;
        }
        return value;
    }

    public static Boolean getBoolean(@NotNull final ResultSet rs, @NotNull final String label) throws SQLException {
        final boolean value = rs.getBoolean(label);
        if (rs.wasNull()) {
            return null;
        }
        return value;
    }

    /**
     * @deprecated must be replaced by {@link ru.yandex.qe.dispenser.domain.dao.SqlUtils#fromJsonb}
     */
    @Deprecated
    public static String getJson(@NotNull final ResultSet rs, @NotNull final String label) throws SQLException {
        final String value = rs.getString(label);
        return StringEscapeUtils.unescapeJava(value.substring(1, value.length() - 1));
    }

    /**
     * @deprecated must be replaced by {@link ru.yandex.qe.dispenser.domain.dao.SqlUtils#fromJsonb}
     */
    @Deprecated
    public static <T> T getJson(@NotNull final ResultSet rs, @NotNull final String label, @NotNull final Class<T> type) throws
            SQLException {
        try {
            return SerializationUtils.readValue(getJson(rs, label), type);
        } catch (IOException e) {
            throw new RuntimeException("Invalid json format", e);
        }
    }

    /**
     * @deprecated must be replaced by {@link ru.yandex.qe.dispenser.domain.dao.SqlUtils#fromJsonb}
     */
    @Deprecated
    public static <T> T getJson(@NotNull final ResultSet rs, @NotNull final String label,
                                @NotNull final TypeReference<T> valueTypeRef) throws
            SQLException {
        try {
            return SerializationUtils.readValue(getJson(rs, label), valueTypeRef);
        } catch (IOException e) {
            throw new RuntimeException("Invalid json format", e);
        }
    }

}
