package ru.yandex.qe.dispenser.domain.dao;

import java.io.IOException;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.util.Arrays;
import java.util.Objects;
import java.util.StringJoiner;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.type.TypeReference;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.postgresql.util.PGobject;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;

import ru.yandex.qe.dispenser.api.util.SerializationUtils;

public enum SqlUtils {
    ;

    @Nullable
    public static Timestamp toTimestamp(@Nullable final Long time) {
        return time != null ? new Timestamp(time) : null;
    }

    @Contract("null -> null")
    public static Long toTime(@Nullable final Timestamp timestamp) {
        return timestamp != null ? timestamp.getTime() : null;
    }

    @NotNull
    public static String toString(@NotNull final Object params) {
        if (params instanceof MapSqlParameterSource) {
            return ((MapSqlParameterSource) params).getValues().toString();
        }
        if (params instanceof Object[]) {
            return Arrays.stream((Object[]) params).map(SqlUtils::toString).collect(Collectors.joining(", ", "[", "]"));
        }
        return params.toString();
    }

    @NotNull
    public static String toString(@NotNull final ResultSet rs) throws SQLException {
        final StringJoiner s = new StringJoiner(", ", "{", "}");
        for (int i = 1; i <= rs.getMetaData().getColumnCount(); i++) {
            s.add(rs.getMetaData().getColumnName(i) + "=" + rs.getString(i));
        }
        return s.toString();
    }

    @NotNull
    public static PGobject toJsonb(@NotNull final Object o) {
        final PGobject jsonb = new PGobject();
        jsonb.setType("jsonb");
        try {
            jsonb.setValue(SerializationUtils.writeValueAsString(o));
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
        return jsonb;
    }

    @NotNull
    public static <T> T fromJsonb(@NotNull final PGobject jsonb, @NotNull final Class<T> valueType) {
        try {
            return SerializationUtils.readValue(jsonb.getValue(), valueType);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }


    @NotNull
    public static <T> T fromJsonb(@NotNull final PGobject jsonb, @NotNull final TypeReference<T> valueType) {
        try {
            return SerializationUtils.readValue(jsonb.getValue(), valueType);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static void checkDeadLocks(@NotNull final SqlParameterSource... batchArgs) {
        final long[] ids = Arrays.stream(batchArgs)
                .map(params -> value(params, "id"))
                .filter(Objects::nonNull)
                .mapToLong(Long.class::cast)
                .toArray();
        for (int i = 0; i < ids.length - 1; i++) {
            if (ids[i] > ids[i + 1]) {
                throw new IllegalStateException("Sort 'id's to prevent deadlocks!");
            }
        }
    }

    @Nullable
    public static Object value(@NotNull final SqlParameterSource source, @NotNull final String paramName) {
        try {
            return source.getValue(paramName);
        } catch (IllegalArgumentException ignored) {
            return null;
        }
    }

    @NotNull
    public static String toSqlType(@NotNull final Class<?> javaType) {
        if (javaType == String.class) {
            return "short_text";
        }
        if (javaType == Long.class) {
            return "bigint";
        }
        throw new IllegalArgumentException("No sql type for '" + javaType.getSimpleName() + "' class!");
    }

    @NotNull
    public static Class<?> fromSqlType(@NotNull final String sqlType) {
        switch (sqlType) {
            case "short_text":
                return String.class;
            case "bigint":
                return Long.class;
            default:
                throw new IllegalArgumentException("No java type for '" + sqlType + "' sql type!");
        }
    }

    public static <T> T fromJsonbNumbersAsStrings(PGobject jsonb, TypeReference<T> valueType) {
        return SerializationUtils.readValueNumbersAsStrings(jsonb.getValue(), valueType);
    }

    public static <T> T fromJsonbNumbersAsStrings(PGobject jsonb, Class<T> valueType) {
        return SerializationUtils.readValueNumbersAsStrings(jsonb.getValue(), valueType);
    }

    public static PGobject toJsonbNumbersAsStrings(Object o) {
        PGobject jsonb = new PGobject();
        jsonb.setType("jsonb");
        try {
            jsonb.setValue(SerializationUtils.writeValueAsStringNumbersAsStrings(o));
        } catch (SQLException e) {
            throw new RuntimeException(e);
        }
        return jsonb;
    }

}
