package ru.yandex.qe.dispenser.domain.dao.base_resources;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.jetbrains.annotations.TestOnly;

import ru.yandex.qe.dispenser.domain.base_resources.BaseResourceChange;

@TestOnly
public class BaseResourceChangeDaoImpl implements BaseResourceChangeDao {

    private final List<BaseResourceChange> collection = new ArrayList<>();

    @Override
    public synchronized BaseResourceChange create(BaseResourceChange.Builder builder) {
        long nextId = collection.stream().max(Comparator.comparing(BaseResourceChange::getId))
                .map(BaseResourceChange::getId).orElse(-1L) + 1L;
        BaseResourceChange result = builder.build(nextId);
        if (collection.stream().anyMatch(t -> t.getQuotaRequestId() == result.getQuotaRequestId()
                && t.getBigOrderId() == result.getBigOrderId()
                && t.getBaseResourceId() == result.getBaseResourceId())) {
            throw new IllegalArgumentException("Conflicting base resource change already exists");
        }
        collection.add(result);
        return result;
    }

    @Override
    public synchronized Set<BaseResourceChange> create(Collection<? extends BaseResourceChange.Builder> builders) {
        if (builders.isEmpty()) {
            return Set.of();
        }
        long nextId = collection.stream().max(Comparator.comparing(BaseResourceChange::getId))
                .map(BaseResourceChange::getId).orElse(-1L) + 1L;
        Set<BaseResourceChange> results = new HashSet<>();
        for (BaseResourceChange.Builder builder : builders) {
            BaseResourceChange result = builder.build(nextId);
            results.add(result);
            nextId++;
        }
        for (BaseResourceChange result : results) {
            if (collection.stream().anyMatch(t -> t.getQuotaRequestId() == result.getQuotaRequestId()
                    && t.getBigOrderId() == result.getBigOrderId()
                    && t.getBaseResourceId() == result.getBaseResourceId())) {
                throw new IllegalArgumentException("Conflicting base resource change already exists");
            }
        }
        collection.addAll(results);
        return results;
    }

    @Override
    public synchronized Optional<BaseResourceChange> getById(long id) {
        return collection.stream().filter(v -> v.getId() == id).findFirst();
    }

    @Override
    public synchronized Set<BaseResourceChange> getByIds(Collection<? extends Long> ids) {
        if (ids.isEmpty()) {
            return Set.of();
        }
        return collection.stream().filter(v -> ids.contains(v.getId())).collect(Collectors.toSet());
    }

    @Override
    public synchronized Set<BaseResourceChange> getByQuotaRequestId(long quotaRequestId) {
        return collection.stream().filter(v -> v.getQuotaRequestId() == quotaRequestId).collect(Collectors.toSet());
    }

    @Override
    public synchronized Set<BaseResourceChange> getByQuotaRequestIds(Collection<? extends Long> quotaRequestIds) {
        if (quotaRequestIds.isEmpty()) {
            return Set.of();
        }
        return collection.stream().filter(v -> quotaRequestIds.contains(v.getQuotaRequestId()))
                .collect(Collectors.toSet());
    }

    @Override
    public synchronized Optional<BaseResourceChange> update(BaseResourceChange.Update update) {
        BaseResourceChange updated = update.build();
        int index = -1;
        for (int i = 0; i < collection.size(); i++) {
            if (collection.get(i).getId() == updated.getId()) {
                index = i;
                break;
            }
        }
        if (index == -1) {
            return Optional.empty();
        }
        collection.remove(index);
        collection.add(updated);
        return Optional.of(updated);
    }

    @Override
    public synchronized Set<BaseResourceChange> update(Collection<? extends BaseResourceChange.Update> updates) {
        if (updates.isEmpty()) {
            return Set.of();
        }
        Set<BaseResourceChange> result = new HashSet<>();
        for (BaseResourceChange.Update update : updates) {
            BaseResourceChange entity = update.build();
            int index = -1;
            for (int i = 0; i < collection.size(); i++) {
                if (collection.get(i).getId() == entity.getId()) {
                    index = i;
                    break;
                }
            }
            if (index == -1) {
                continue;
            }
            collection.remove(index);
            collection.add(entity);
            result.add(entity);
        }
        return result;
    }

    @Override
    public synchronized Optional<BaseResourceChange> deleteById(long id) {
        int index = -1;
        for (int i = 0; i < collection.size(); i++) {
            if (collection.get(i).getId() == id) {
                index = i;
                break;
            }
        }
        if (index == -1) {
            return Optional.empty();
        }
        BaseResourceChange removed = collection.get(index);
        collection.remove(index);
        return Optional.of(removed);
    }

    @Override
    public synchronized Set<BaseResourceChange> deleteByIds(Collection<? extends Long> ids) {
        if (ids.isEmpty()) {
            return Set.of();
        }
        Set<BaseResourceChange> result = new HashSet<>();
        for (long id : ids) {
            int index = -1;
            for (int i = 0; i < collection.size(); i++) {
                if (collection.get(i).getId() == id) {
                    index = i;
                    break;
                }
            }
            if (index == -1) {
                continue;
            }
            result.add(collection.get(index));
            collection.remove(index);
        }
        return result;
    }

    @Override
    public synchronized void clear() {
        collection.clear();
    }

}
