package ru.yandex.qe.dispenser.domain.dao.base_resources;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.domain.base_resources.BaseResourceLimit;
import ru.yandex.qe.dispenser.domain.base_resources.CampaignLimits;

@Component
public class PerCampaignLimitsCache {

    private final BaseResourceLimitDao baseResourceLimitDao;
    private final LoadingCache<Long, CampaignLimits> cache;

    public PerCampaignLimitsCache(BaseResourceLimitDao baseResourceLimitDao) {
        this.baseResourceLimitDao = baseResourceLimitDao;
        this.cache = CacheBuilder.newBuilder()
                .expireAfterWrite(5, TimeUnit.MINUTES)
                .expireAfterAccess(24, TimeUnit.HOURS)
                .maximumSize(10000)
                .build(new Loader(baseResourceLimitDao));
    }

    public CampaignLimits getByCampaignId(long id) {
        return CacheRefreshScheduler.wrapException(() -> cache.get(id));
    }

    public Set<CampaignLimits> getByCampaignIds(Collection<? extends Long> ids) {
        return CacheRefreshScheduler.wrapException(() -> new HashSet<>(cache.getAll(ids).values()));
    }

    public void clear() {
        cache.invalidateAll();
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public void refresh() {
        List<Long> keysToRefresh = cache.asMap().keySet().stream().distinct().collect(Collectors.toList());
        Set<BaseResourceLimit> refreshed = Lists.partition(keysToRefresh, 500).stream()
                .flatMap(keys -> baseResourceLimitDao.getByCampaigns(keys).stream())
                .collect(Collectors.toSet());
        Map<Long, Set<BaseResourceLimit>> refreshedByCampaign = refreshed.stream()
                .collect(Collectors.groupingBy(BaseResourceLimit::getCampaignId, Collectors.toSet()));
        Map<Long, CampaignLimits> refreshResult = new HashMap<>();
        keysToRefresh.forEach(key -> refreshResult.put(key, toCampaignLimits(key,
                refreshedByCampaign.getOrDefault(key, Set.of()))));
        cache.putAll(refreshResult);
    }

    private static final class Loader extends CacheLoader<Long, CampaignLimits> {

        private final BaseResourceLimitDao baseResourceLimitDao;

        private Loader(BaseResourceLimitDao baseResourceLimitDao) {
            this.baseResourceLimitDao = baseResourceLimitDao;
        }

        @Override
        public CampaignLimits load(Long key) {
            return toCampaignLimits(key, baseResourceLimitDao.getByCampaign(key));
        }

        @Override
        public Map<Long, CampaignLimits> loadAll(Iterable<? extends Long> keys) {
            List<Long> keySet = StreamSupport.stream(keys.spliterator(), false).distinct()
                    .collect(Collectors.toList());
            Set<BaseResourceLimit> limits = Lists.partition(keySet, 500).stream()
                    .flatMap(page -> baseResourceLimitDao.getByCampaigns(page).stream()).collect(Collectors.toSet());
            Map<Long, Set<BaseResourceLimit>> limitsByCampaign = limits.stream()
                    .collect(Collectors.groupingBy(BaseResourceLimit::getCampaignId, Collectors.toSet()));
            Map<Long, CampaignLimits> result = new HashMap<>();
            keySet.forEach(campaignId -> result.put(campaignId,
                    toCampaignLimits(campaignId, limitsByCampaign.getOrDefault(campaignId, Set.of()))));
            return result;
        }

    }

    private static CampaignLimits toCampaignLimits(long campaignId, Set<BaseResourceLimit> limits) {
        if (limits.isEmpty()) {
            return new CampaignLimits(campaignId, Map.of());
        }
        Map<Long, Set<BaseResourceLimit>> limitsByBaseResourceId = limits.stream()
                .collect(Collectors.groupingBy(BaseResourceLimit::getBaseResourceId, Collectors.toSet()));
        return new CampaignLimits(campaignId, limitsByBaseResourceId);
    }

}
