package ru.yandex.qe.dispenser.domain.dao.base_resources;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import com.google.common.base.Functions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.domain.base_resources.BaseResourceMapping;
import ru.yandex.qe.dispenser.domain.base_resources.BigOrderMappings;
import ru.yandex.qe.dispenser.domain.base_resources.CampaignMappings;
import ru.yandex.qe.dispenser.domain.base_resources.ResourceAndSegmentsId;

@Component
public class PerCampaignMappingsCache {

    private final BaseResourceMappingDao baseResourceMappingDao;
    private final LoadingCache<Long, CampaignMappings> cache;

    public PerCampaignMappingsCache(BaseResourceMappingDao baseResourceMappingDao) {
        this.baseResourceMappingDao = baseResourceMappingDao;
        this.cache = CacheBuilder.newBuilder()
                .expireAfterWrite(5, TimeUnit.MINUTES)
                .expireAfterAccess(24, TimeUnit.HOURS)
                .maximumSize(10000)
                .build(new Loader(baseResourceMappingDao));
    }

    public CampaignMappings getByCampaignId(long id) {
        return CacheRefreshScheduler.wrapException(() -> cache.get(id));
    }

    public Set<CampaignMappings> getByCampaignIds(Collection<? extends Long> ids) {
        return CacheRefreshScheduler.wrapException(() -> new HashSet<>(cache.getAll(ids).values()));
    }

    public void clear() {
        cache.invalidateAll();
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public void refresh() {
        List<Long> keysToRefresh = cache.asMap().keySet().stream().distinct().collect(Collectors.toList());
        Set<BaseResourceMapping> refreshed = Lists.partition(keysToRefresh, 500).stream()
                .flatMap(keys -> baseResourceMappingDao.getByCampaigns(keys).stream())
                .collect(Collectors.toSet());
        Map<Long, Set<BaseResourceMapping>> refreshedByCampaign = refreshed.stream()
                .collect(Collectors.groupingBy(BaseResourceMapping::getCampaignId, Collectors.toSet()));
        Map<Long, CampaignMappings> refreshResult = new HashMap<>();
        keysToRefresh.forEach(key -> refreshResult.put(key, toCampaignMappings(key,
                refreshedByCampaign.getOrDefault(key, Set.of()))));
        cache.putAll(refreshResult);
    }

    private static final class Loader extends CacheLoader<Long, CampaignMappings> {

        private final BaseResourceMappingDao baseResourceMappingDao;

        private Loader(BaseResourceMappingDao baseResourceMappingDao) {
            this.baseResourceMappingDao = baseResourceMappingDao;
        }

        @Override
        public CampaignMappings load(Long key) {
            return toCampaignMappings(key, baseResourceMappingDao.getByCampaign(key));
        }

        @Override
        public Map<Long, CampaignMappings> loadAll(Iterable<? extends Long> keys) {
            List<Long> keySet = StreamSupport.stream(keys.spliterator(), false).distinct()
                    .collect(Collectors.toList());
            Set<BaseResourceMapping> mappings = Lists.partition(keySet, 500).stream()
                    .flatMap(page -> baseResourceMappingDao.getByCampaigns(page).stream()).collect(Collectors.toSet());
            Map<Long, Set<BaseResourceMapping>> mappingsByCampaign = mappings.stream()
                    .collect(Collectors.groupingBy(BaseResourceMapping::getCampaignId, Collectors.toSet()));
            Map<Long, CampaignMappings> result = new HashMap<>();
            keySet.forEach(campaignId -> result.put(campaignId,
                    toCampaignMappings(campaignId, mappingsByCampaign.getOrDefault(campaignId, Set.of()))));
            return result;
        }

    }

    private static CampaignMappings toCampaignMappings(long campaignId, Set<BaseResourceMapping> mappings) {
        if (mappings.isEmpty()) {
            return new CampaignMappings(campaignId, Map.of());
        }
        Map<Long, Set<BaseResourceMapping>> mappingsByBigOrder = mappings.stream()
                .collect(Collectors.groupingBy(BaseResourceMapping::getBigOrderId, Collectors.toSet()));
        Map<Long, BigOrderMappings> perBigOrderMappings = new HashMap<>();
        mappingsByBigOrder.forEach((bigOrderId, bigOrderMappings) -> {
            Map<Long, BaseResourceMapping> mappingsById = bigOrderMappings.stream()
                    .collect(Collectors.toMap(BaseResourceMapping::getId, Functions.identity()));
            Map<Long, Set<Long>> mappingIdsByBaseResourceId = bigOrderMappings.stream()
                    .collect(Collectors.groupingBy(BaseResourceMapping::getBaseResourceId,
                            Collectors.mapping(BaseResourceMapping::getId, Collectors.toSet())));
            Map<ResourceAndSegmentsId, Set<Long>> mappingIdsByResourceId = new HashMap<>();
            bigOrderMappings.forEach(mapping -> mapping.getRelation().getAllResources().forEach(resource ->
                    mappingIdsByResourceId.computeIfAbsent(resource, k -> new HashSet<>()).add(mapping.getId())));
            perBigOrderMappings.put(bigOrderId, new BigOrderMappings(bigOrderId, mappingsById,
                    mappingIdsByBaseResourceId, mappingIdsByResourceId));
        });
        return new CampaignMappings(campaignId, perBigOrderMappings);
    }

}
