package ru.yandex.qe.dispenser.domain.dao.base_resources;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Lists;
import org.postgresql.util.PGobject;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.domain.base_resources.BaseResourceChange;
import ru.yandex.qe.dispenser.domain.base_resources.BaseResourceChangeByService;
import ru.yandex.qe.dispenser.domain.dao.SqlDaoBase;
import ru.yandex.qe.dispenser.domain.dao.SqlUtils;

public class SqlBaseResourceChangeDao extends SqlDaoBase implements BaseResourceChangeDao {

    private static final String INSERT_QUERY = "INSERT INTO quota_request_base_change" +
            " (quota_request_id, big_order_id, base_resource_id, amount, amount_by_service)" +
            " VALUES (:quotaRequestId, :bigOrderId, :baseResourceId, :amount, :amountByService) RETURNING id";
    private static final String GET_BY_ID = "SELECT id, quota_request_id, big_order_id," +
            " base_resource_id, amount, amount_by_service FROM quota_request_base_change WHERE" +
            " id = :id";
    private static final String GET_BY_IDS = "SELECT id, quota_request_id, big_order_id," +
            " base_resource_id, amount, amount_by_service FROM quota_request_base_change WHERE" +
            " id IN (:ids)";
    private static final String GET_BY_QUOTA_REQUEST_ID = "SELECT id, quota_request_id, big_order_id," +
            " base_resource_id, amount, amount_by_service FROM quota_request_base_change WHERE" +
            " quota_request_id = :quotaRequestId";
    private static final String GET_BY_QUOTA_REQUEST_IDS = "SELECT id, quota_request_id, big_order_id," +
            " base_resource_id, amount, amount_by_service FROM quota_request_base_change WHERE" +
            " quota_request_id IN (:quotaRequestIds)";
    private static final String UPDATE_BY_ID = "UPDATE quota_request_base_change SET amount = :amount," +
            " amount_by_service = :amountByService WHERE id = :id RETURNING id, quota_request_id, big_order_id," +
            " base_resource_id, amount, amount_by_service";
    private static final String DELETE_BY_ID = "DELETE FROM quota_request_base_change WHERE id = :id RETURNING id," +
            " quota_request_id, big_order_id, base_resource_id, amount, amount_by_service";
    private static final String DELETE_BY_IDS = "DELETE FROM quota_request_base_change WHERE id IN (:ids) RETURNING" +
            " id, quota_request_id, big_order_id, base_resource_id, amount, amount_by_service";
    private static final String CLEAR_QUERY = "TRUNCATE quota_request_base_change";

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public BaseResourceChange create(BaseResourceChange.Builder builder) {
        try {
            long id = jdbcTemplate.queryForObject(INSERT_QUERY, toInsertParams(builder), Long.class);
            return builder.build(id);
        } catch (DuplicateKeyException e) {
            throw new IllegalArgumentException("Conflicting base resource change already exists", e);
        }
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<BaseResourceChange> create(Collection<? extends BaseResourceChange.Builder> builders) {
        if (builders.isEmpty()) {
            return Set.of();
        }
        Set<BaseResourceChange> result = new HashSet<>();
        List<List<BaseResourceChange.Builder>> pages = Lists.partition(new ArrayList<>(builders), 100);
        for (List<BaseResourceChange.Builder> page : pages) {
            Map<String, Object> params = new HashMap<>();
            StringBuilder queryBuilder = new StringBuilder();
            queryBuilder
                    .append("INSERT INTO quota_request_base_change")
                    .append(" (quota_request_id, big_order_id, base_resource_id, amount, amount_by_service)")
                    .append(" VALUES ");
            for (int i = 0; i < page.size(); i++) {
                queryBuilder
                        .append("(")
                        .append(":quotaRequestId")
                        .append(i)
                        .append(", ")
                        .append(":bigOrderId")
                        .append(i)
                        .append(", ")
                        .append(":baseResourceId")
                        .append(i)
                        .append(", ")
                        .append(":amount")
                        .append(i)
                        .append(", ")
                        .append(":amountByService")
                        .append(i)
                        .append(")");
                if (i < page.size() - 1) {
                    queryBuilder.append(", ");
                }
                toInsertParams(params, page.get(i), i);
            }
            queryBuilder.append(" RETURNING id, quota_request_id, big_order_id," +
                    "base_resource_id, amount, amount_by_service");
            result.addAll(jdbcTemplate.queryForSet(queryBuilder.toString(), params, this::toModel));
        }
        return result;
    }

    @Override
    public Optional<BaseResourceChange> getById(long id) {
        return jdbcTemplate.queryForOptional(GET_BY_ID, Map.of("id", id), this::toModel);
    }

    @Override
    public Set<BaseResourceChange> getByIds(Collection<? extends Long> ids) {
        if (ids.isEmpty()) {
            return Set.of();
        }
        List<List<Long>> pages = Lists.partition(new ArrayList<>(ids), 500);
        return pages.stream().flatMap(page -> jdbcTemplate.queryForSet(GET_BY_IDS, Map.of("ids", page),
                this::toModel).stream()).collect(Collectors.toSet());
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<BaseResourceChange> getByQuotaRequestId(long quotaRequestId) {
        return jdbcTemplate.queryForSet(GET_BY_QUOTA_REQUEST_ID,
                Map.of("quotaRequestId", quotaRequestId), this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<BaseResourceChange> getByQuotaRequestIds(Collection<? extends Long> quotaRequestIds) {
        if (quotaRequestIds.isEmpty()) {
            return Set.of();
        }
        List<List<Long>> pages = Lists.partition(new ArrayList<>(quotaRequestIds), 500);
        return pages.stream().flatMap(page -> jdbcTemplate.queryForSet(GET_BY_QUOTA_REQUEST_IDS,
                Map.of("quotaRequestIds", page), this::toModel).stream()).collect(Collectors.toSet());
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Optional<BaseResourceChange> update(BaseResourceChange.Update update) {
        BaseResourceChange updated = update.build();
        Map<String, ?> updateParams = toUpdateParams(updated);
        return jdbcTemplate.queryForOptional(UPDATE_BY_ID, updateParams, this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<BaseResourceChange> update(Collection<? extends BaseResourceChange.Update> updates) {
        if (updates.isEmpty()) {
            return Set.of();
        }
        Set<BaseResourceChange> result = new HashSet<>();
        List<List<BaseResourceChange.Update>> pages = Lists.partition(new ArrayList<>(updates), 100);
        for (List<BaseResourceChange.Update> page : pages) {
            Map<String, Object> params = new HashMap<>();
            StringBuilder queryBuilder = new StringBuilder();
            queryBuilder
                    .append("UPDATE quota_request_base_change AS original SET")
                    .append(" amount = updated.update_amount, amount_by_service = updated.update_amount_by_service")
                    .append(" FROM (VALUES ");
            for (int i = 0; i < page.size(); i++) {
                queryBuilder
                        .append("(")
                        .append(":id")
                        .append(i)
                        .append(", ")
                        .append(":amount")
                        .append(i)
                        .append(", ")
                        .append(":amountByService")
                        .append(i)
                        .append(")");
                if (i < page.size() - 1) {
                    queryBuilder.append(", ");
                }
                toUpdateParams(params, page.get(i).build(), i);
            }
            queryBuilder.append(") AS updated (update_id, update_amount, update_amount_by_service) WHERE" +
                    " updated.update_id = original.id");
            queryBuilder.append(" RETURNING id, quota_request_id, big_order_id," +
                    "base_resource_id, amount, amount_by_service");
            result.addAll(jdbcTemplate.queryForSet(queryBuilder.toString(), params, this::toModel));
        }
        return result;
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Optional<BaseResourceChange> deleteById(long id) {
        return jdbcTemplate.queryForOptional(DELETE_BY_ID, Map.of("id", id), this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<BaseResourceChange> deleteByIds(Collection<? extends Long> ids) {
        if (ids.isEmpty()) {
            return Set.of();
        }
        List<List<Long>> pages = Lists.partition(new ArrayList<>(ids), 500);
        return pages.stream().flatMap(page -> jdbcTemplate.queryForSet(DELETE_BY_IDS, Map.of("ids", page),
                this::toModel).stream()).collect(Collectors.toSet());
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public void clear() {
        jdbcTemplate.update(CLEAR_QUERY);
    }

    private Map<String, ?> toInsertParams(BaseResourceChange.Builder builder) {
        return Map.of(
                "quotaRequestId", builder.getQuotaRequestId()
                        .orElseThrow(() -> new NoSuchElementException("Quota request id is required")),
                "bigOrderId", builder.getBigOrderId()
                        .orElseThrow(() -> new NoSuchElementException("Big order id is required")),
                "baseResourceId", builder.getBaseResourceId()
                        .orElseThrow(() -> new NoSuchElementException("Base resource id is required")),
                "amount", builder.getAmount()
                        .orElseThrow(() -> new NoSuchElementException("Amount is required")),
                "amountByService", SqlUtils.toJsonbNumbersAsStrings(builder.getAmountByService()
                        .orElseThrow(() -> new NoSuchElementException("Amount by service is required")))
        );
    }

    private void toInsertParams(Map<String, Object> params, BaseResourceChange.Builder builder, int i) {
        params.put("quotaRequestId" + i, builder.getQuotaRequestId()
                .orElseThrow(() -> new NoSuchElementException("Quota request id is required")));
        params.put("bigOrderId" + i, builder.getBigOrderId()
                        .orElseThrow(() -> new NoSuchElementException("Big order id is required")));
        params.put("baseResourceId" + i, builder.getBaseResourceId()
                        .orElseThrow(() -> new NoSuchElementException("Base resource id is required")));
        params.put("amount" + i, builder.getAmount()
                        .orElseThrow(() -> new NoSuchElementException("Amount is required")));
        params.put("amountByService" + i, SqlUtils.toJsonbNumbersAsStrings(builder.getAmountByService()
                        .orElseThrow(() -> new NoSuchElementException("Amount by service is required"))));
    }

    private Map<String, ?> toUpdateParams(BaseResourceChange updated) {
        return Map.of(
                "amount", updated.getAmount(),
                "amountByService", SqlUtils.toJsonbNumbersAsStrings(updated.getAmountByService()),
                "id", updated.getId()
        );
    }

    private void toUpdateParams(Map<String, Object> params, BaseResourceChange updated, int i) {
        params.put("amount" + i, updated.getAmount());
        params.put("amountByService" + i, SqlUtils.toJsonbNumbersAsStrings(updated.getAmountByService()));
        params.put("id" + i, updated.getId());
    }

    private BaseResourceChange toModel(ResultSet rs, int i) throws SQLException {
        return BaseResourceChange.builder()
                .id(rs.getLong("id"))
                .quotaRequestId(rs.getLong("quota_request_id"))
                .bigOrderId(rs.getLong("big_order_id"))
                .baseResourceId(rs.getLong("base_resource_id"))
                .amount(rs.getLong("amount"))
                .amountByService(SqlUtils.fromJsonbNumbersAsStrings((PGobject) rs.getObject("amount_by_service"),
                        BaseResourceChangeByService.class))
                .build();
    }

}
