package ru.yandex.qe.dispenser.domain.dao.base_resources;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.Set;

import org.springframework.dao.DuplicateKeyException;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.domain.base_resources.BaseResource;
import ru.yandex.qe.dispenser.domain.dao.SqlDaoBase;

public class SqlBaseResourceDao extends SqlDaoBase implements BaseResourceDao {

    private static final String INSERT_QUERY = "INSERT INTO base_resource" +
            " (key, name, base_resource_type_id, segment_ids)" +
            " VALUES (:key, :name, :baseResourceTypeId, :segmentIds) RETURNING id";
    private static final String GET_BY_ID_QUERY = "SELECT id, key, name, base_resource_type_id, segment_ids FROM" +
            " base_resource WHERE id = :id";
    private static final String GET_BY_IDS_QUERY = "SELECT id, key, name, base_resource_type_id, segment_ids FROM" +
            " base_resource WHERE id IN (:ids)";
    private static final String GET_FIRST_PAGE_QUERY = "SELECT id, key, name, base_resource_type_id, segment_ids FROM" +
            " base_resource ORDER BY id ASC LIMIT :limit";
    private static final String GET_PAGE_QUERY = "SELECT id, key, name, base_resource_type_id, segment_ids FROM" +
            " base_resource WHERE id > :idFrom ORDER BY id ASC LIMIT :limit";
    private static final String GET_BY_BASE_RESOURCE_TYPE_IDS_QUERY = "SELECT id, key, name, base_resource_type_id," +
            " segment_ids FROM base_resource WHERE base_resource_type_id IN (:ids)";
    private static final String GET_BY_KEY_QUERY = "SELECT id, key, name, base_resource_type_id, segment_ids FROM" +
            " base_resource WHERE key = :key";
    private static final String GET_BY_KEYS_QUERY = "SELECT id, key, name, base_resource_type_id, segment_ids FROM" +
            " base_resource WHERE key IN (:keys)";
    private static final String CLEAR_QUERY = "TRUNCATE base_resource CASCADE";

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public BaseResource create(BaseResource.Builder builder) {
        try {
            long id = jdbcTemplate.queryForObject(INSERT_QUERY, toInsertParams(builder), Long.class);
            return builder.build(id);
        } catch (DuplicateKeyException e) {
            throw new IllegalArgumentException("Conflicting base resource already exists", e);
        }
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Optional<BaseResource> getById(long id) {
        return jdbcTemplate.queryForOptional(GET_BY_ID_QUERY, Map.of("id", id), this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<BaseResource> getByIds(Collection<? extends Long> ids) {
        if (ids.isEmpty()) {
            return Set.of();
        }
        return jdbcTemplate.queryForSet(GET_BY_IDS_QUERY, Map.of("ids", ids), this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public List<BaseResource> getPage(Long idFrom, int size) {
        return Optional.ofNullable(idFrom).map(from -> jdbcTemplate.query(GET_PAGE_QUERY,
                        Map.of("idFrom", from, "limit", (long) size), this::toModel))
                .orElseGet(() -> jdbcTemplate.query(GET_FIRST_PAGE_QUERY,
                        Map.of("limit", (long) size), this::toModel));
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<BaseResource> getByBaseResourceTypeIds(Collection<? extends Long> ids) {
        if (ids.isEmpty()) {
            return Set.of();
        }
        return jdbcTemplate.queryForSet(GET_BY_BASE_RESOURCE_TYPE_IDS_QUERY, Map.of("ids", ids), this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Optional<BaseResource> getByKey(String key) {
        return jdbcTemplate.queryForOptional(GET_BY_KEY_QUERY, Map.of("key", key), this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<BaseResource> getByKeys(Collection<? extends String> keys) {
        if (keys.isEmpty()) {
            return Set.of();
        }
        return jdbcTemplate.queryForSet(GET_BY_KEYS_QUERY, Map.of("keys", keys), this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public void clear() {
        jdbcTemplate.update(CLEAR_QUERY);
    }

    private Map<String, ?> toInsertParams(BaseResource.Builder builder) {
        return Map.of(
                "key", builder.getKey().orElseThrow(() -> new NoSuchElementException("Key is required")),
                "name", builder.getName().orElseThrow(() -> new NoSuchElementException("Name is required")),
                "baseResourceTypeId", builder.getBaseResourceTypeId()
                        .orElseThrow(() -> new NoSuchElementException("Base resource type id is required")),
                "segmentIds", builder.getSegmentIds().stream().mapToLong(v -> v).toArray()
        );
    }

    private BaseResource toModel(ResultSet rs, int i) throws SQLException {
        return BaseResource.builder()
                .id(rs.getLong("id"))
                .key(rs.getString("key"))
                .name(rs.getString("name"))
                .baseResourceTypeId(rs.getLong("base_resource_type_id"))
                .addSegmentIds((Long[]) rs.getArray("segment_ids").getArray())
                .build();
    }

}
