package ru.yandex.qe.dispenser.domain.dao.base_resources;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.Set;

import org.springframework.dao.DuplicateKeyException;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.domain.base_resources.BaseResourceLimit;
import ru.yandex.qe.dispenser.domain.dao.SqlDaoBase;

public class SqlBaseResourceLimitDao extends SqlDaoBase implements BaseResourceLimitDao {

    private static final String INSERT_QUERY = "INSERT INTO base_resource_limit" +
            " (base_resource_id, campaign_id, limit_value)" +
            " VALUES (:baseResourceId, :campaignId, :limitValue) RETURNING id";
    private static final String GET_BY_ID_QUERY = "SELECT id, base_resource_id, campaign_id," +
            " limit_value FROM base_resource_limit WHERE id = :id";
    private static final String GET_BY_IDS_QUERY = "SELECT id, base_resource_id, campaign_id," +
            " limit_value FROM base_resource_limit WHERE id IN (:ids)";
    private static final String GET_FIRST_PAGE_QUERY = "SELECT id, base_resource_id, campaign_id," +
            " limit_value FROM base_resource_limit ORDER BY id ASC LIMIT :limit";
    private static final String GET_PAGE_QUERY = "SELECT id, base_resource_id, campaign_id," +
            " limit_value FROM base_resource_limit WHERE id > :idFrom ORDER BY id ASC LIMIT :limit";
    private static final String GET_BY_CAMPAIGN_ID_QUERY = "SELECT id, base_resource_id, campaign_id," +
            " limit_value FROM base_resource_limit WHERE campaign_id = :campaignId";
    private static final String GET_BY_CAMPAIGN_IDS_QUERY = "SELECT id, base_resource_id, campaign_id," +
            " limit_value FROM base_resource_limit WHERE campaign_id IN (:campaignIds)";
    private static final String UPDATE_BY_ID = "UPDATE base_resource_limit SET limit_value = :limitValue" +
            " WHERE id = :id RETURNING id, base_resource_id, campaign_id, limit_value";
    private static final String DELETE_BY_ID = "DELETE FROM base_resource_limit WHERE id = :id RETURNING id," +
            " base_resource_id, campaign_id, limit_value";
    private static final String DELETE_BY_IDS = "DELETE FROM base_resource_limit WHERE id IN (:ids) RETURNING id," +
            " base_resource_id, campaign_id, limit_value";
    private static final String GET_BY_KEY_QUERY = "SELECT id, base_resource_id, campaign_id," +
            " limit_value FROM base_resource_limit WHERE base_resource_id = :baseResourceId" +
            " AND campaign_id = :campaignId";
    private static final String GET_BY_CAMPAIGN_ID_AND_BASE_RESOURCE_IDS_QUERY = "SELECT id, base_resource_id," +
            " campaign_id, limit_value FROM base_resource_limit WHERE campaign_id = :campaignId" +
            " AND base_resource_id in (:baseResourceIds)";
    private static final String CLEAR_QUERY = "TRUNCATE base_resource_limit";

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public BaseResourceLimit create(BaseResourceLimit.Builder builder) {
        try {
            long id = jdbcTemplate.queryForObject(INSERT_QUERY, toInsertParams(builder), Long.class);
            return builder.build(id);
        } catch (DuplicateKeyException e) {
            throw new IllegalArgumentException("Conflicting base resource limit already exists", e);
        }
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<BaseResourceLimit> create(Collection<? extends BaseResourceLimit.Builder> builders) {
        if (builders.isEmpty()) {
            return Set.of();
        }
        Map<String, Object> params = new HashMap<>();
        StringBuilder queryBuilder = new StringBuilder();
        queryBuilder
                .append("INSERT INTO base_resource_limit")
                .append(" (base_resource_id, campaign_id, limit_value)")
                .append(" VALUES ");
        List<BaseResourceLimit.Builder> list = new ArrayList<>(builders);
        for (int i = 0; i < list.size(); i++) {
            queryBuilder
                    .append("(")
                    .append(":baseResourceId")
                    .append(i)
                    .append(", ")
                    .append(":campaignId")
                    .append(i)
                    .append(", ")
                    .append(":limitValue")
                    .append(i)
                    .append(")");
            if (i < builders.size() - 1) {
                queryBuilder.append(", ");
            }
            toInsertParams(params, list.get(i), i);
        }
        queryBuilder.append(" RETURNING id, base_resource_id, campaign_id, limit_value");
        return jdbcTemplate.queryForSet(queryBuilder.toString(), params, this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Optional<BaseResourceLimit> getById(long id) {
        return jdbcTemplate.queryForOptional(GET_BY_ID_QUERY, Map.of("id", id), this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<BaseResourceLimit> getByIds(Collection<? extends Long> ids) {
        if (ids.isEmpty()) {
            return Set.of();
        }
        return jdbcTemplate.queryForSet(GET_BY_IDS_QUERY, Map.of("ids", ids), this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public List<BaseResourceLimit> getPage(Long idFrom, int size) {
        return Optional.ofNullable(idFrom).map(from -> jdbcTemplate.query(GET_PAGE_QUERY,
                Map.of("idFrom", from, "limit", (long) size), this::toModel))
                .orElseGet(() -> jdbcTemplate.query(GET_FIRST_PAGE_QUERY,
                        Map.of("limit", (long) size), this::toModel));
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<BaseResourceLimit> getByCampaign(long campaignId) {
        return jdbcTemplate.queryForSet(GET_BY_CAMPAIGN_ID_QUERY, Map.of("campaignId", campaignId), this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<BaseResourceLimit> getByCampaigns(Collection<? extends Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return Set.of();
        }
        return jdbcTemplate.queryForSet(GET_BY_CAMPAIGN_IDS_QUERY, Map.of("campaignIds", campaignIds), this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Optional<BaseResourceLimit> update(BaseResourceLimit.Update update) {
        BaseResourceLimit updated = update.build();
        Map<String, ?> updateParams = toUpdateParams(updated);
        return jdbcTemplate.queryForOptional(UPDATE_BY_ID, updateParams, this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<BaseResourceLimit> update(Collection<? extends BaseResourceLimit.Update> updates) {
        if (updates.isEmpty()) {
            return Set.of();
        }
        Map<String, Object> params = new HashMap<>();
        StringBuilder queryBuilder = new StringBuilder();
        queryBuilder
                .append("UPDATE base_resource_limit AS original SET")
                .append(" limit_value = updated.update_limit_value")
                .append(" FROM (VALUES ");
        List<BaseResourceLimit.Update> list = new ArrayList<>(updates);
        for (int i = 0; i < list.size(); i++) {
            queryBuilder
                    .append("(")
                    .append(":id")
                    .append(i)
                    .append(", ")
                    .append(":limitValue")
                    .append(i)
                    .append(")");
            if (i < list.size() - 1) {
                queryBuilder.append(", ");
            }
            toUpdateParams(params, list.get(i).build(), i);
        }
        queryBuilder.append(") AS updated (update_id, update_limit_value) WHERE" +
                " updated.update_id = original.id");
        queryBuilder.append(" RETURNING id, base_resource_id, campaign_id, limit_value");
        return jdbcTemplate.queryForSet(queryBuilder.toString(), params, this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Optional<BaseResourceLimit> deleteById(long id) {
        return jdbcTemplate.queryForOptional(DELETE_BY_ID, Map.of("id", id), this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<BaseResourceLimit> deleteByIds(Collection<? extends Long> ids) {
        if (ids.isEmpty()) {
            return Set.of();
        }
        return jdbcTemplate.queryForSet(DELETE_BY_IDS, Map.of("ids", ids), this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Optional<BaseResourceLimit> getByKey(BaseResourceLimit.Key key) {
        return jdbcTemplate.queryForOptional(GET_BY_KEY_QUERY,
                Map.of("baseResourceId", key.getBaseResourceId(),
                        "campaignId", key.getCampaignId()),
                this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<BaseResourceLimit> getByCampaignAndBaseResources(long campaignId,
                                                                Collection<? extends Long> baseResourceIds) {
        if (baseResourceIds.isEmpty()) {
            return Set.of();
        }
        return jdbcTemplate.queryForSet(GET_BY_CAMPAIGN_ID_AND_BASE_RESOURCE_IDS_QUERY,
                Map.of("campaignId", campaignId, "baseResourceIds", baseResourceIds), this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public void clear() {
        jdbcTemplate.update(CLEAR_QUERY);
    }

    private Map<String, ?> toInsertParams(BaseResourceLimit.Builder builder) {
        return Map.of(
                "baseResourceId", builder.getBaseResourceId()
                        .orElseThrow(() -> new NoSuchElementException("Base resource id is required")),
                "campaignId", builder.getCampaignId()
                        .orElseThrow(() -> new NoSuchElementException("Campaign id is required")),
                "limitValue", builder.getLimit()
                        .orElseThrow(() -> new NoSuchElementException("Limit is required"))
        );
    }

    private void toInsertParams(Map<String, Object> params, BaseResourceLimit.Builder builder, int index) {
        params.put("baseResourceId" + index, builder.getBaseResourceId()
                .orElseThrow(() -> new NoSuchElementException("Base resource id is required")));
        params.put("campaignId" + index, builder.getCampaignId()
                .orElseThrow(() -> new NoSuchElementException("Campaign id is required")));
        params.put("limitValue" + index, builder.getLimit()
                .orElseThrow(() -> new NoSuchElementException("Limit is required")));
    }

    private Map<String, ?> toUpdateParams(BaseResourceLimit updated) {
        return Map.of(
                "limitValue", updated.getLimit(),
                "id", updated.getId()
        );
    }

    private void toUpdateParams(Map<String, Object> params, BaseResourceLimit updated, int index) {
        params.put("limitValue" + index, updated.getLimit());
        params.put("id" + index, updated.getId());
    }

    private BaseResourceLimit toModel(ResultSet rs, int i) throws SQLException {
        return BaseResourceLimit.builder()
                .id(rs.getLong("id"))
                .baseResourceId(rs.getLong("base_resource_id"))
                .campaignId(rs.getLong("campaign_id"))
                .limit(rs.getLong("limit_value"))
                .build();
    }

}
