package ru.yandex.qe.dispenser.domain.dao.base_resources;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.Set;

import org.postgresql.util.PGobject;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.domain.base_resources.BaseResourceMapping;
import ru.yandex.qe.dispenser.domain.base_resources.BaseResourceRelation;
import ru.yandex.qe.dispenser.domain.dao.SqlDaoBase;
import ru.yandex.qe.dispenser.domain.dao.SqlUtils;

public class SqlBaseResourceMappingDao extends SqlDaoBase implements BaseResourceMappingDao {

    private static final String INSERT_QUERY = "INSERT INTO base_resource_mapping" +
            " (base_resource_id, campaign_id, big_order_id, service_id, relation)" +
            " VALUES (:baseResourceId, :campaignId, :bigOrderId, :serviceId, :relation) RETURNING id";
    private static final String GET_BY_ID_QUERY = "SELECT id, base_resource_id, campaign_id, big_order_id," +
            " service_id, relation FROM base_resource_mapping WHERE id = :id";
    private static final String GET_BY_IDS_QUERY = "SELECT id, base_resource_id, campaign_id, big_order_id," +
            " service_id, relation FROM base_resource_mapping WHERE id IN (:ids)";
    private static final String GET_FIRST_PAGE_QUERY = "SELECT id, base_resource_id, campaign_id, big_order_id," +
            " service_id, relation FROM base_resource_mapping ORDER BY id ASC LIMIT :limit";
    private static final String GET_PAGE_QUERY = "SELECT id, base_resource_id, campaign_id, big_order_id," +
            " service_id, relation FROM base_resource_mapping WHERE id > :idFrom ORDER BY id ASC LIMIT :limit";
    private static final String GET_BY_CAMPAIGN_ID_QUERY = "SELECT id, base_resource_id, campaign_id, big_order_id," +
            " service_id, relation FROM base_resource_mapping WHERE campaign_id = :campaignId";
    private static final String GET_BY_CAMPAIGN_IDS_QUERY = "SELECT id, base_resource_id, campaign_id, big_order_id," +
            " service_id, relation FROM base_resource_mapping WHERE campaign_id IN (:campaignIds)";
    private static final String GET_MAPPED_BASE_RESOURCE_IDS_QUERY = "SELECT DISTINCT(base_resource_id)" +
            " AS mapped_base_resource_id FROM base_resource_mapping WHERE campaign_id = :campaignId" +
            " AND base_resource_id IN (:baseResourceIds)";
    private static final String CLEAR_QUERY = "TRUNCATE base_resource_mapping";

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public BaseResourceMapping create(BaseResourceMapping.Builder builder) {
        try {
            long id = jdbcTemplate.queryForObject(INSERT_QUERY, toInsertParams(builder), Long.class);
            return builder.build(id);
        } catch (DuplicateKeyException e) {
            throw new IllegalArgumentException("Conflicting base resource mapping already exists", e);
        }
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Optional<BaseResourceMapping> getById(long id) {
        return jdbcTemplate.queryForOptional(GET_BY_ID_QUERY, Map.of("id", id), this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<BaseResourceMapping> getByIds(Collection<? extends Long> ids) {
        if (ids.isEmpty()) {
            return Set.of();
        }
        return jdbcTemplate.queryForSet(GET_BY_IDS_QUERY, Map.of("ids", ids), this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public List<BaseResourceMapping> getPage(Long idFrom, int size) {
        return Optional.ofNullable(idFrom).map(from -> jdbcTemplate.query(GET_PAGE_QUERY,
                Map.of("idFrom", from, "limit", (long) size), this::toModel))
                .orElseGet(() -> jdbcTemplate.query(GET_FIRST_PAGE_QUERY,
                        Map.of("limit", (long) size), this::toModel));
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<BaseResourceMapping> getByCampaign(long campaignId) {
        return jdbcTemplate.queryForSet(GET_BY_CAMPAIGN_ID_QUERY, Map.of("campaignId", campaignId), this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<BaseResourceMapping> getByCampaigns(Collection<? extends Long> campaignIds) {
        if (campaignIds.isEmpty()) {
            return Set.of();
        }
        return jdbcTemplate.queryForSet(GET_BY_CAMPAIGN_IDS_QUERY, Map.of("campaignIds", campaignIds), this::toModel);
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<Long> getMappedBaseResourceIds(long campaignId, Collection<? extends Long> baseResourceIds) {
        if (baseResourceIds.isEmpty()) {
            return Set.of();
        }
        return jdbcTemplate.queryForSet(GET_MAPPED_BASE_RESOURCE_IDS_QUERY,
                Map.of("campaignId", campaignId, "baseResourceIds", baseResourceIds),
                (rs, i) -> rs.getLong("mapped_base_resource_id"));
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public void clear() {
        jdbcTemplate.update(CLEAR_QUERY);
    }

    private Map<String, ?> toInsertParams(BaseResourceMapping.Builder builder) {
        return Map.of(
                "baseResourceId", builder.getBaseResourceId()
                        .orElseThrow(() -> new NoSuchElementException("Base resource id is required")),
                "campaignId", builder.getCampaignId()
                        .orElseThrow(() -> new NoSuchElementException("Campaign id is required")),
                "bigOrderId", builder.getBigOrderId()
                        .orElseThrow(() -> new NoSuchElementException("Big order id is required")),
                "serviceId", builder.getServiceId()
                        .orElseThrow(() -> new NoSuchElementException("Service id is required")),
                "relation", SqlUtils.toJsonbNumbersAsStrings(builder.getRelation()
                        .orElseThrow(() -> new NoSuchElementException("Relation is required")))
        );
    }

    private BaseResourceMapping toModel(ResultSet rs, int i) throws SQLException {
        return BaseResourceMapping.builder()
                .id(rs.getLong("id"))
                .baseResourceId(rs.getLong("base_resource_id"))
                .campaignId(rs.getLong("campaign_id"))
                .bigOrderId(rs.getLong("big_order_id"))
                .serviceId(rs.getLong("service_id"))
                .relation(SqlUtils.fromJsonbNumbersAsStrings((PGobject) rs.getObject("relation"),
                        BaseResourceRelation.class))
                .build();
    }

}
