package ru.yandex.qe.dispenser.domain.dao.bot;

import java.util.Objects;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import ru.yandex.qe.dispenser.api.v1.DiProject;
import ru.yandex.qe.dispenser.api.v1.DiService;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;

import static java.util.Objects.requireNonNull;

@ParametersAreNonnullByDefault
public class MappedPreOrder extends LongIndexBase implements PreOrderBase {
    @NotNull
    private final Key key;
    @Nullable
    private final String name;
    @Nullable
    private final Long storageQuantity;
    @Nullable
    private final Long serverQuantity;
    @NotNull
    private final Status status;
    private double reserveRate;

    private MappedPreOrder(final Builder builder) {
        key = new Key(builder);

        serverQuantity = builder.serverQuantity;
        storageQuantity = builder.storageQuantity;
        status = builder.status;
        name = builder.name;
        reserveRate = builder.reserveRate;

        if (builder.id != null) {
            setId(builder.id);
        }
    }

    @NotNull
    public Key getKey() {
        return key;
    }

    @Nullable
    public String getName() {
        return name;
    }

    @NotNull
    public Project getProject() {
        return key.project;
    }

    @NotNull
    public Service getService() {
        return key.service;
    }

    @NotNull
    public Long getBigOrderId() {
        return key.bigOrderId;
    }

    @Override
    @Nullable
    public Long getServerId() {
        return key.serverId;
    }

    @Override
    @Nullable
    public Long getServerQuantity() {
        return serverQuantity;
    }

    @Override
    @Nullable
    public Long getStorageId() {
        return key.storageId;
    }

    @Override
    @Nullable
    public Long getStorageQuantity() {
        return storageQuantity;
    }

    @NotNull
    public Long getBigOrderConfigId() {
        return key.bigOrderConfigId;
    }

    @NotNull
    public Status getStatus() {
        return status;
    }

    @NotNull
    public String getGroupKey() {
        return key.groupKey;
    }

    @NotNull
    public Long getCampaignGroupId() {
        return key.campaignGroupId;
    }

    public double getReserveRate() {
        return reserveRate;
    }

    public void setReserveRate(final double reserveRate) {
        this.reserveRate = reserveRate;
    }

    @NotNull
    public Builder copyBuilder() {
        return new Builder()
                .id(getId())
                .key(key)
                .serverQuantity(serverQuantity)
                .storageQuantity(storageQuantity)
                .status(status)
                .name(name)
                .reserveRate(reserveRate);
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final MappedPreOrder that = (MappedPreOrder) o;
        return key.equals(that.key) &&
                Objects.equals(storageQuantity, that.storageQuantity) &&
                Objects.equals(serverQuantity, that.serverQuantity) &&
                Objects.equals(reserveRate, that.reserveRate);
    }

    @Override
    public int hashCode() {
        return Objects.hash(key, storageQuantity, serverQuantity);
    }

    @Override
    public String toString() {
        return "MappedPreOrder{" +
                "key=" + key +
                ", name='" + name + '\'' +
                ", storageQuantity=" + storageQuantity +
                ", serverQuantity=" + serverQuantity +
                ", status=" + status +
                "} " + super.toString();
    }

    @NotNull
    public MappedPreOrder.View toView() {
        return new View(this);
    }

    public enum Status {
        DRAFTED,
        ORDERED
    }

    public static class Key {
        @NotNull
        private final Project project;
        @NotNull
        private final Service service;
        @NotNull
        private final Long bigOrderId;
        @NotNull
        private final Long bigOrderConfigId;
        @Nullable
        private final Long serverId;
        @Nullable
        private final Long storageId;
        @NotNull
        private final String groupKey;
        @NotNull
        private final Long campaignGroupId;

        public Key(@NotNull final Project project,
                   @NotNull final Service service,
                   @NotNull final Long bigOrderId,
                   @NotNull final Long bigOrderConfigId,
                   @Nullable final Long serverId,
                   @Nullable final Long storageId,
                   @NotNull final String groupKey,
                   @NotNull final Long campaignGroupId) {
            this.project = project;
            this.service = service;
            this.bigOrderId = bigOrderId;
            this.bigOrderConfigId = bigOrderConfigId;
            this.serverId = serverId;
            this.storageId = storageId;
            this.groupKey = groupKey;
            this.campaignGroupId = campaignGroupId;
        }

        private Key(final Builder builder) {
            project = requireNonNull(builder.project);
            service = requireNonNull(builder.service);
            bigOrderId = requireNonNull(builder.bigOrderId);
            bigOrderConfigId = requireNonNull(builder.bigOrderConfigId);
            serverId = builder.serverId;
            storageId = builder.storageId;
            groupKey = requireNonNull(builder.groupKey);
            campaignGroupId = requireNonNull(builder.campaignGroupId);
        }

        public @NotNull Project getProject() {
            return project;
        }

        public @NotNull Service getService() {
            return service;
        }

        public @NotNull Long getBigOrderId() {
            return bigOrderId;
        }

        public @NotNull Long getBigOrderConfigId() {
            return bigOrderConfigId;
        }

        public @Nullable Long getServerId() {
            return serverId;
        }

        public @Nullable Long getStorageId() {
            return storageId;
        }

        public @NotNull String getGroupKey() {
            return groupKey;
        }

        public @NotNull Long getCampaignGroupId() {
            return campaignGroupId;
        }

        @SuppressWarnings("OverlyComplexMethod")
        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final Key key = (Key) o;
            return project.equals(key.project) &&
                    service.equals(key.service) &&
                    bigOrderId.equals(key.bigOrderId) &&
                    bigOrderConfigId.equals(key.bigOrderConfigId) &&
                    Objects.equals(serverId, key.serverId) &&
                    Objects.equals(storageId, key.storageId) &&
                    Objects.equals(groupKey, key.groupKey) &&
                    Objects.equals(campaignGroupId, key.campaignGroupId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(project, service, bigOrderId, bigOrderConfigId, serverId, storageId, groupKey, campaignGroupId);
        }

        @Override
        public String toString() {
            return "Key{" +
                    "project=" + project +
                    ", service=" + service +
                    ", bigOrderId=" + bigOrderId +
                    ", bigOrderConfigId=" + bigOrderConfigId +
                    ", serverId=" + serverId +
                    ", storageId=" + storageId +
                    ", groupKey=" + groupKey +
                    ", campaignGroupId=" + campaignGroupId +
                    '}';
        }
    }

    public static class Builder {
        @Nullable
        private Long id;
        @Nullable
        private Project project;
        @Nullable
        private Service service;
        @Nullable
        private Long bigOrderId;
        @Nullable
        private Long bigOrderConfigId;
        @Nullable
        private Long serverId;
        @Nullable
        private Long serverQuantity;
        @Nullable
        private Long storageId;
        @Nullable
        private Long storageQuantity;
        @NotNull
        private Status status = Status.DRAFTED;
        @Nullable
        private String groupKey;
        @Nullable
        private Long campaignGroupId;
        @Nullable
        private String name;
        public double reserveRate;

        public Builder key(final Key key) {
            this.project = key.project;
            this.service = key.service;
            this.bigOrderId = key.bigOrderId;
            this.bigOrderConfigId = key.bigOrderConfigId;
            this.serverId = key.serverId;
            this.storageId = key.storageId;
            this.groupKey = key.groupKey;
            this.campaignGroupId = key.campaignGroupId;
            return this;
        }

        public Builder id(final Long id) {
            this.id = id;
            return this;
        }

        public Builder project(final Project project) {
            this.project = project;
            return this;
        }

        public Builder service(final Service service) {
            this.service = service;
            return this;
        }

        public Builder bigOrderId(final Long bigOrderId) {
            this.bigOrderId = bigOrderId;
            return this;
        }

        public Builder bigOrderConfigId(final Long bigOrderConfigId) {
            this.bigOrderConfigId = bigOrderConfigId;
            return this;
        }

        public Builder serverId(@Nullable final Long serverId) {
            this.serverId = serverId;
            return this;
        }

        public Builder serverQuantity(@Nullable final Long serverCount) {
            this.serverQuantity = serverCount;
            return this;
        }

        public Builder storageId(@Nullable final Long storageId) {
            this.storageId = storageId;
            return this;
        }

        public Builder storageQuantity(@Nullable final Long storageCount) {
            this.storageQuantity = storageCount;
            return this;
        }

        public Builder status(@NotNull final Status status) {
            this.status = status;
            return this;
        }

        public Builder groupKey(@NotNull final String groupKey) {
            this.groupKey = groupKey;
            return this;
        }

        public Builder name(@Nullable final String name) {
            this.name = name;
            return this;
        }

        public Builder reserveRate(final double reserveRate) {
            this.reserveRate = reserveRate;
            return this;
        }

        public Builder campaignGroupId(final long campaignGroupId) {
            this.campaignGroupId = campaignGroupId;
            return this;
        }

        public MappedPreOrder build() {
            return new MappedPreOrder(this);
        }
    }

    public static class View {
        @NotNull
        private final Long id;
        @Nullable
        private final Long storageQuantity;
        @Nullable
        private final Long serverQuantity;
        @NotNull
        private final DiProject project;
        @NotNull
        private final DiService service;
        @NotNull
        private final Long bigOrderId;
        @NotNull
        private final Long bigOrderConfigId;
        @Nullable
        private final Long serverId;
        @Nullable
        private final Long storageId;
        @NotNull
        private final Long campaignGroupId;

        @SuppressWarnings("ConstructorWithTooManyParameters")
        @JsonCreator
        public View(@JsonProperty("id") @NotNull final Long id,
                    @JsonProperty("storageQuantity") @Nullable final Long storageQuantity,
                    @JsonProperty("serverQuantity") @Nullable final Long serverQuantity,
                    @JsonProperty("project") @NotNull final DiProject project,
                    @JsonProperty("service") @NotNull final DiService service,
                    @JsonProperty("bigOrderId") @NotNull final Long bigOrderId,
                    @JsonProperty("bigOrderConfigId") @NotNull final Long bigOrderConfigId,
                    @JsonProperty("serverId") @Nullable final Long serverId,
                    @JsonProperty("storageId") @Nullable final Long storageId,
                    @JsonProperty("campaignGroupId") @NotNull final Long campaignGroupId) {
            this.id = id;
            this.storageQuantity = storageQuantity;
            this.serverQuantity = serverQuantity;
            this.project = project;
            this.service = service;
            this.bigOrderId = bigOrderId;
            this.bigOrderConfigId = bigOrderConfigId;
            this.serverId = serverId;
            this.storageId = storageId;
            this.campaignGroupId = campaignGroupId;
        }

        private View(final MappedPreOrder order) {
            id = order.getId();
            project = order.getProject().toView();
            service = order.getService().toView();
            bigOrderId = order.getBigOrderId();
            bigOrderConfigId = order.getBigOrderConfigId();
            serverId = order.getServerId();
            serverQuantity = order.getServerQuantity();
            storageId = order.getStorageId();
            storageQuantity = order.getStorageQuantity();
            campaignGroupId = order.getCampaignGroupId();
        }

        @NotNull
        public Long getId() {
            return id;
        }

        @Nullable
        public Long getStorageQuantity() {
            return storageQuantity;
        }

        @Nullable
        public Long getServerQuantity() {
            return serverQuantity;
        }

        @NotNull
        public DiProject getProject() {
            return project;
        }

        @NotNull
        public DiService getService() {
            return service;
        }

        @NotNull
        public Long getBigOrderId() {
            return bigOrderId;
        }

        @NotNull
        public Long getBigOrderConfigId() {
            return bigOrderConfigId;
        }

        @Nullable
        public Long getServerId() {
            return serverId;
        }

        @Nullable
        public Long getStorageId() {
            return storageId;
        }

        @NotNull
        public Long getCampaignGroupId() {
            return campaignGroupId;
        }
    }

}
