package ru.yandex.qe.dispenser.domain.dao.bot;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.domain.dao.SqlDaoBase;
import ru.yandex.qe.dispenser.domain.index.LongIndexable;

public abstract class SqlSyncableBotDao<T extends LongIndexable> extends SqlDaoBase implements SyncableBotDao<T> {
    private final String getAllQuery;
    private final String getAllIdQuery;
    private final String deleteQuery;
    private final String clearQuery;
    private final String insertQuery;
    private final String updateQuery;
    private final String upsertQuery;
    private final String getByIdQuery;

    public SqlSyncableBotDao(final String tableName, final Map<String, String> fields) {
        getAllQuery = "SELECT * FROM " + tableName;
        getAllIdQuery = "SELECT id FROM " + tableName;
        deleteQuery = "UPDATE " + tableName + " SET deleted = TRUE WHERE id in (:id) AND deleted != TRUE";
        clearQuery = "TRUNCATE " + tableName + " CASCADE";
        final String fieldNames = StringUtils.join(fields.keySet(), ", ");
        final String values = StringUtils.join(fields.values(), ", ");
        insertQuery = String.format("INSERT INTO %s (id, %s) VALUES (:id, %s)", tableName, fieldNames, values);

        final String sets = "SET " + fields.entrySet().stream()
                .map(entry -> entry.getKey() + " = " + entry.getValue())
                .collect(Collectors.joining(", "));


        updateQuery = String.format("UPDATE %s %s WHERE id = :id", tableName, sets);
        upsertQuery = String.format("INSERT INTO %s (id, %s) VALUES(:id, %s) ON CONFLICT (id) DO UPDATE %s", tableName, fieldNames, values, sets);
        getByIdQuery = getAllQuery + " WHERE id in (:id)";
    }

    @NotNull
    @Override
    @Transactional
    public T create(@NotNull final T configuration) {
        jdbcTemplate.update(insertQuery, toParams(configuration));
        return configuration;
    }

    @Override
    @Transactional
    public boolean update(@NotNull final T configuration) {
        return jdbcTemplate.update(updateQuery, toParams(configuration)) > 0;
    }

    @Override
    @Transactional
    public void upsert(@NotNull final List<T> items) {
        if (items.isEmpty()) {
            return;
        }
        final MapSqlParameterSource[] mapSqlParameterSources = items.stream()
                .map(this::toParams)
                .map(MapSqlParameterSource::new)
                .toArray(MapSqlParameterSource[]::new);

        jdbcTemplate.batchUpdate(upsertQuery, mapSqlParameterSources);
    }


    @Override
    @Transactional
    public boolean clear() {
        return jdbcTemplate.update(clearQuery) > 0;
    }

    @NotNull
    @Override
    public Set<T> getAll() {
        return jdbcTemplate.queryForSet(getAllQuery, this::toItem);
    }

    @NotNull
    @Override
    public Set<Long> getAllIds() {
        return jdbcTemplate.queryForSet(getAllIdQuery, (rs, num) -> rs.getLong("id"));
    }

    @NotNull
    @Override
    public T read(final Long id) {
        return jdbcTemplate.queryForObject(getByIdQuery, Collections.singletonMap("id", id), this::toItem);
    }

    @NotNull
    @Override
    public Set<T> readAll(final Collection<Long> ids) {
        if (ids.isEmpty()) {
            return Collections.emptySet();
        }
        return jdbcTemplate.queryForSet(getByIdQuery, Collections.singletonMap("id", ids), this::toItem);
    }

    @Override
    @Transactional
    public void deleteByIds(final Set<Long> toRemoveIds) {
        jdbcTemplate.update(deleteQuery, Collections.singletonMap("id", toRemoveIds));
    }

    protected abstract T toItem(final ResultSet rs, final int rowNum) throws SQLException;

    protected abstract Map<String, ?> toParams(final T configuration);
}
