package ru.yandex.qe.dispenser.domain.dao.bot.bigorder;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import javax.inject.Inject;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.domain.bot.BigOrder;
import ru.yandex.qe.dispenser.domain.dao.base_resources.CacheRefreshScheduler;

@Component
public class BigOrderCache {

    private final BigOrderDao bigOrderDao;
    private final LoadingCache<Long, Optional<BigOrder>> cache;

    @Inject
    public BigOrderCache(BigOrderDao bigOrderDao) {
        this.bigOrderDao = bigOrderDao;
        this.cache = CacheBuilder.newBuilder()
                .expireAfterWrite(5, TimeUnit.MINUTES)
                .expireAfterAccess(24, TimeUnit.HOURS)
                .maximumSize(10000)
                .build(new Loader(bigOrderDao));
    }

    public Optional<BigOrder> getById(long id) {
        return CacheRefreshScheduler.wrapException(() -> cache.get(id));
    }

    public Set<BigOrder> getByIds(Collection<? extends Long> ids) {
        return CacheRefreshScheduler.wrapException(() -> cache.getAll(ids).values()
                .stream().flatMap(Optional::stream).collect(Collectors.toSet()));
    }

    public void clear() {
        cache.invalidateAll();
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public void refresh() {
        List<Long> keysToRefresh = cache.asMap().keySet().stream().distinct().collect(Collectors.toList());
        Map<Long, BigOrder> refreshed = Lists.partition(keysToRefresh, 500).stream()
                .flatMap(keys -> bigOrderDao.getByIds(keys).stream())
                .collect(Collectors.toMap(BigOrder::getId, Function.identity()));
        Map<Long, Optional<BigOrder>> refreshedById = new HashMap<>();
        keysToRefresh.forEach(key -> refreshedById.put(key, Optional.ofNullable(refreshed.get(key))));
        cache.putAll(refreshedById);
    }

    private static final class Loader extends CacheLoader<Long, Optional<BigOrder>> {

        private final BigOrderDao botBigOrderDao;

        private Loader(BigOrderDao botBigOrderDao) {
            this.botBigOrderDao = botBigOrderDao;
        }

        @Override
        public Optional<BigOrder> load(Long key) {
            return botBigOrderDao.getByIds(Set.of(key)).stream().findFirst();
        }

        @Override
        public Map<Long, Optional<BigOrder>> loadAll(Iterable<? extends Long> keys) {
            List<Long> keySet = StreamSupport.stream(keys.spliterator(), false).distinct()
                    .collect(Collectors.toList());
            Map<Long, BigOrder> knownEntitiesByKey = Lists.partition(keySet, 500).stream()
                    .flatMap(page -> botBigOrderDao.getByIds(page).stream())
                    .collect(Collectors.toMap(BigOrder::getId, Function.identity()));
            Map<Long, Optional<BigOrder>> result = new HashMap<>();
            keySet.forEach(key -> result.put(key, Optional.ofNullable(knownEntitiesByKey.get(key))));
            return result;
        }

    }

}
