package ru.yandex.qe.dispenser.domain.dao.bot.bigorder

import com.fasterxml.jackson.core.type.TypeReference
import org.postgresql.util.PGobject
import org.springframework.dao.DuplicateKeyException
import org.springframework.transaction.annotation.Propagation
import org.springframework.transaction.annotation.Transactional
import ru.yandex.qe.dispenser.domain.bot.BigOrder
import ru.yandex.qe.dispenser.domain.bot.BigOrderConfig
import ru.yandex.qe.dispenser.domain.dao.SqlDaoBase
import ru.yandex.qe.dispenser.domain.dao.SqlUtils
import java.sql.ResultSet
import java.sql.SQLException
import java.time.LocalDate
import java.time.format.DateTimeFormatter

open class SqlBigOrderDao: SqlDaoBase(), BigOrderDao {

    @Transactional(propagation = Propagation.MANDATORY)
    override fun getById(id: Long): BigOrder? {
        return jdbcTemplate.queryForOptional(
            "SELECT * FROM bot_big_order WHERE id = :id",
            mapOf("id" to id)
        ) { rs, rowNum -> toItem(rs, rowNum) }.orElse(null)
    }

    @Transactional(propagation = Propagation.MANDATORY)
    override fun getByIds(ids: Collection<Long>): Set<BigOrder> {
        if (ids.isEmpty()) {
            return emptySet()
        }
        return jdbcTemplate.queryForSet(
            "SELECT * FROM bot_big_order WHERE id IN (:ids)",
            mapOf("ids" to ids)
        ) { rs, rowNum -> toItem(rs, rowNum) }
    }

    @Transactional(propagation = Propagation.MANDATORY)
    override fun getPage(idFrom: Long?, limit: Long): List<BigOrder> {
        return if (idFrom == null) {
            jdbcTemplate.query(
                "SELECT * FROM bot_big_order WHERE deleted = FALSE ORDER BY id ASC LIMIT :limit",
                mapOf("limit" to limit)
            ) { rs, i -> toItem(rs, i) }
        } else {
            jdbcTemplate.query(
                "SELECT * FROM bot_big_order WHERE deleted = FALSE AND id > :idFrom ORDER BY id ASC LIMIT :limit",
                mapOf("limit" to limit, "idFrom" to idFrom)
            ) { rs, i -> toItem(rs, i) }
        }
    }

    @Transactional(propagation = Propagation.MANDATORY)
    override fun getAll(): Set<BigOrder> {
        return jdbcTemplate.queryForSet(
            "SELECT * FROM bot_big_order",
        ) { rs, rowNum -> toItem(rs, rowNum) }
    }

    @Transactional(propagation = Propagation.MANDATORY)
    override fun create(builder: BigOrder.Builder): BigOrder {
        return try {
            val id = jdbcTemplate.queryForObject(
                "INSERT INTO bot_big_order (date, configs, deleted) VALUES (:date, :configs, :deleted) RETURNING id",
                toInsertParams(builder), Long::class.java)
            builder.build(id)
        } catch (e: DuplicateKeyException) {
            throw IllegalArgumentException("Conflicting big order already exists", e)
        }
    }

    @Transactional(propagation = Propagation.MANDATORY)
    override fun update(update: BigOrder.Update): BigOrder {
        val updated = update.build()
        return jdbcTemplate.queryForOptional(
            "UPDATE bot_big_order SET date = :date, configs = :configs, deleted = :deleted WHERE id = :id RETURNING id, date, configs, deleted",
            toUpdateParams(updated)
        ){ rs, i -> toItem(rs, i) }.orElseThrow { IllegalArgumentException("No big order with id ${updated.id}") }
    }

    @Transactional(propagation = Propagation.MANDATORY)
    override fun clear(): Boolean {
        return jdbcTemplate.update("TRUNCATE bot_big_order CASCADE") > 0
    }

    @Throws(SQLException::class)
    private fun toItem(rs: ResultSet, rowNum: Int): BigOrder {
        return BigOrder.builder(LocalDate.parse(rs.getString("date"), DateTimeFormatter.ISO_LOCAL_DATE))
            .configs(SqlUtils.fromJsonb(rs.getObject("configs") as PGobject, CONFIG_TYPE))
            .deleted(rs.getBoolean("deleted"))
            .build(rs.getLong("id"))
    }

    private fun toInsertParams(builder: BigOrder.Builder): Map<String, *> {
        return mapOf(
            "date" to builder.date.format(DateTimeFormatter.ISO_LOCAL_DATE),
            "configs" to SqlUtils.toJsonb(builder.configs),
            "deleted" to builder.isDeleted
        )
    }

    private fun toUpdateParams(builder: BigOrder): Map<String, *> {
        return mapOf(
            "id" to builder.id,
            "date" to builder.date.format(DateTimeFormatter.ISO_LOCAL_DATE),
            "configs" to SqlUtils.toJsonb(builder.configs),
            "deleted" to builder.isDeleted
        )
    }

    companion object {
        @JvmField
        val CONFIG_TYPE: TypeReference<List<BigOrderConfig>> = object : TypeReference<List<BigOrderConfig>>() { }
    }

}
