package ru.yandex.qe.dispenser.domain.dao.bot.configuration;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import com.google.common.collect.ImmutableMap;

import ru.yandex.qe.dispenser.api.util.SerializationUtils;
import ru.yandex.qe.dispenser.domain.bot.Configuration;
import ru.yandex.qe.dispenser.domain.bot.ConfigurationWithComponents;
import ru.yandex.qe.dispenser.domain.dao.bot.BotConfiguration;
import ru.yandex.qe.dispenser.domain.dao.bot.CompleteBotConfiguration;
import ru.yandex.qe.dispenser.domain.dao.bot.SimpleBotConfiguration;
import ru.yandex.qe.dispenser.domain.dao.bot.SqlSyncableBotDao;

public class SqlBotConfigurationDao extends SqlSyncableBotDao<BotConfiguration<? extends Configuration>>
        implements BotConfigurationDao {

    private static final String GET_CONFIGURATION_IN_PRE_ORDERS_BY = "SELECT bc.* FROM bot_configuration bc " +
            "JOIN bot_synced_pre_order bpo ON bc.id = bpo.server_id OR bc.id = bpo.storage_id " +
            "WHERE bc.with_components = true AND bpo.big_order_id IN (:big_order_id)";

    public SqlBotConfigurationDao() {
        super("bot_configuration", ImmutableMap.of(
                "json", "to_jsonb(:json)",
                "with_components", ":withComponents",
                "price", ":price",
                "resources_components_price", ":resourcesComponentsPrice",
                "deleted", ":deleted"
        ));
    }

    @Override
    protected BotConfiguration<? extends Configuration> toItem(final ResultSet rs, final int rowNum) throws SQLException {
        final boolean withComponents = rs.getBoolean("with_components");

        if (withComponents) {
            return new CompleteBotConfiguration(getJson(rs, "json", ConfigurationWithComponents.class),
                    rs.getLong("price"),
                    rs.getLong("resources_components_price"),
                    rs.getBoolean("deleted"));
        }

        return new SimpleBotConfiguration(getJson(rs, "json", Configuration.class), rs.getBoolean("deleted"));
    }

    @Override
    protected Map<String, ?> toParams(final BotConfiguration<? extends Configuration> botConfiguration) {
        final Configuration configuration = botConfiguration.getConfiguration();
        return ImmutableMap.<String, Object>builder()
                .put("id", configuration.getId())
                .put("json", SerializationUtils.writeValueAsString(configuration))
                .put("withComponents", botConfiguration.isWithComponents())
                .put("price", botConfiguration.getPrice())
                .put("resourcesComponentsPrice", botConfiguration.getResourceComponentsPrice())
                .put("deleted", botConfiguration.isDeletedInBot())
                .build();
    }

    @Override
    public Map<Long, CompleteBotConfiguration> getConfigurationsInBigOrders(final Set<Long> bigOrderIds) {
        if (bigOrderIds.isEmpty()) {
            return Collections.emptyMap();
        }

        final Map<Long, CompleteBotConfiguration> result = new HashMap<>();

        jdbcTemplate.query(GET_CONFIGURATION_IN_PRE_ORDERS_BY, Collections.singletonMap("big_order_id", bigOrderIds), (rs) -> {
            final CompleteBotConfiguration botConfiguration = (CompleteBotConfiguration) toItem(rs, -1);
            result.put(botConfiguration.getId(), botConfiguration);
        });

        return result;
    }
}
