package ru.yandex.qe.dispenser.domain.dao.bot.preorder;

import java.time.format.DateTimeFormatter;
import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.DuplicateKeyException;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.datasource.lookup.DataSourceLookupFailureException;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.bot.BigOrder;
import ru.yandex.qe.dispenser.domain.bot.BigOrderConfig;
import ru.yandex.qe.dispenser.domain.dao.InMemoryLongKeyDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.bot.MappedPreOrder;
import ru.yandex.qe.dispenser.domain.dao.bot.SimplePreOrder;
import ru.yandex.qe.dispenser.domain.dao.bot.bigorder.BigOrderDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.bot.configuration.BotConfigurationDaoImpl;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;

public class MappedPreOrderDaoImpl extends InMemoryLongKeyDaoImpl<MappedPreOrder> implements MappedPreOrderDao {

    private static final String ALLOWED_TYPE_CODE = "forecast";
    private static final String NEW_CATEGORY_CODE = "new";

    @Autowired
    private BigOrderDaoImpl bigOrderDao;
    @Autowired
    private BotConfigurationDaoImpl configurationDao;

    private final Set<MappedPreOrder.Key> unique = new HashSet<>();


    @Override
    public void createAll(final Collection<MappedPreOrder> preOrders) {
        for (final MappedPreOrder preOrder : preOrders) {
            create(preOrder);
        }
    }

    @Override
    public Optional<MappedPreOrder> readById(final long id) {
        return filter(po -> po.getId() == id).findFirst();
    }

    @NotNull
    @Override
    public MappedPreOrder create(final @NotNull MappedPreOrder newInstance) {
        final long id = newInstance.getId();
        if (unique.contains(newInstance.getKey())) {
            throw new DuplicateKeyException(
                    newInstance.getClass().getSimpleName() + " object with key " + newInstance.getKey() + " already exists!");
        }
        if (id2obj.put(id, newInstance) != null) {
            throw new DuplicateKeyException("'" + newInstance.getClass().getSimpleName() + "' object with id = [" + id + "] already exists!");
        }
        unique.add(newInstance.getKey());
        return newInstance;
    }

    @Override
    public boolean delete(final @NotNull MappedPreOrder obj) {
        final long id = obj.getId();
        if (id2obj.remove(id) == null) {
            throw new DataSourceLookupFailureException("'" + obj.getClass().getSimpleName() + "' with id = [" + id + "] not exists to delete!");
        }
        unique.remove(obj.getKey());
        return true;
    }

    @Override
    public Set<MappedPreOrder> readByProjectAndServiceAndBigOrders(final Project project, final Set<Service> services,
                                                                   final Collection<Long> bigOrderIds, final long campaignGroupId) {
        final Set<Long> bigOrderIdsSet = new HashSet<>(bigOrderIds);
        return filter(o -> o.getProject().equals(project)
                && bigOrderIdsSet.contains(o.getBigOrderId())
                && services.contains(o.getService())
                && o.getCampaignGroupId().equals(campaignGroupId))
                .collect(Collectors.toSet());
    }

    @Override
    public boolean updateAll(final @NotNull Collection<MappedPreOrder> preOrders) {
        return super.updateAll(preOrders);
    }

    @Override
    public boolean update(final @NotNull MappedPreOrder obj) {
        final long id = obj.getId();
        if (id2obj.computeIfPresent(id, (pk, t) -> obj) == null) {
            throw new IllegalArgumentException(obj.getClass().getSimpleName() + " object not exists to update!");
        }
        return true;
    }

    @Override
    public boolean deleteAll(final @NotNull Collection<MappedPreOrder> preOrders) {
        return super.deleteAll(preOrders);
    }

    @Override
    public void lockForChanges() {
    }

    @Override
    public Set<Long> getOrdersIdsInStatus(final Set<Long> preOrderIdsToCheck, final MappedPreOrder.Status status) {
        return filter(r -> preOrderIdsToCheck.contains(r.getId()) && r.getStatus() == status)
                .map(LongIndexBase::getId)
                .collect(Collectors.toSet());
    }

    @Override
    public boolean hasPreOrdersInCampaignGroup(final long campaignGroupId) {
        return getAll().stream().anyMatch(o -> Long.valueOf(campaignGroupId).equals(o.getCampaignGroupId()));
    }

    @Override
    public boolean hasPreOrdersInCampaignGroupForOrdersOtherThan(final long campaignGroupId, final Set<Long> orderIds) {
        return getAll().stream().anyMatch(o -> Long.valueOf(campaignGroupId).equals(o.getCampaignGroupId()) && !orderIds.contains(o.getBigOrderId()));
    }

    @Override
    public boolean clear() {
        final boolean result = id2obj.size() > 0;
        id2obj.clear();
        unique.clear();
        return result;
    }

    @Override
    public Collection<SimplePreOrder> readSimplePreOrders(final Set<Long> ids) {
        return filter(po -> ids.contains(po.getId()))
                .map(p -> {
                    final BigOrder bigOrder = bigOrderDao.getById(p.getBigOrderId());
                    if (bigOrder == null) {
                        throw new EmptyResultDataAccessException("No big order with id " + p.getBigOrderId(), 1);
                    }
                    String bigOrderLocation = null;
                    for (final Map.Entry<String, Long> entry : getNewConfigIdByLocation(bigOrder).entrySet()) {
                        if (p.getBigOrderConfigId().equals(entry.getValue())) {
                            bigOrderLocation = entry.getKey();
                            break;
                        }
                    }
                    String configurationName = null;
                    if (p.getServerId() != null) {
                        configurationName = configurationDao.read(p.getServerId()).getConfiguration().getFullName();
                    }

                    return new SimplePreOrder(
                            p.getId(),
                            p.getService(),
                            configurationName,
                            bigOrder.getDate().format(DateTimeFormatter.ISO_LOCAL_DATE),
                            bigOrderLocation
                    );
                })
                .collect(Collectors.toList());
    }

    @Override
    public Map<Long, MappedPreOrder> getPreOrdersByBigOrderIds(final Set<Long> bigOrderIds) {
        return filter(po -> bigOrderIds.contains(po.getBigOrderId()))
                .collect(Collectors.toMap(LongIndexBase::getId, Function.identity()));
    }

    @NotNull
    private Map<String, Long> getNewConfigIdByLocation(BigOrder bigOrder) {
        return getNewForecastConfigByLocationKey(bigOrder, NEW_CATEGORY_CODE);
    }

    @NotNull
    private Map<String, Long> getNewForecastConfigByLocationKey(BigOrder bigOrder, String categoryCode) {
        final Map<String, List<BigOrderConfig>> forecastConfigByLocationKey = bigOrder.getConfigs().stream()
                .filter(config -> ALLOWED_TYPE_CODE.equals(config.getTypeCode()) && categoryCode.equals(config.getCategoryCode()))
                .collect(Collectors.groupingBy(BigOrderConfig::getLocationKey));
        Map<String, Long> result = new HashMap<>();
        forecastConfigByLocationKey
                .forEach((key, configs) -> selectConfig(configs).ifPresent(c -> result.put(key, c.getId())));
        return result;
    }

    @NotNull
    private Optional<BigOrderConfig> selectConfig(Collection<BigOrderConfig> configsCollection) {
        return configsCollection.stream().max(Comparator.comparing(BigOrderConfig::getId));
    }

}
