package ru.yandex.qe.dispenser.domain.dao.bot.preorder;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;

import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.SqlDaoBase;
import ru.yandex.qe.dispenser.domain.dao.bot.MappedPreOrder;
import ru.yandex.qe.dispenser.domain.dao.bot.SimplePreOrder;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;

import static ru.yandex.qe.dispenser.domain.util.CollectionUtils.ids;

public class SqlMappedPreOrderDao extends SqlDaoBase implements MappedPreOrderDao {

    private static final String GET_ORDERS_BY_PROJECT_AND_SERVICE_AND_BIG_ORDER_QUERY = "SELECT * FROM bot_pre_order WHERE project_id = :projectId AND bot_big_order_id IN (:bigOrderIds) AND service_id IN (:serviceId) AND bot_campaign_group_id = :campaignGroupId";
    private static final String CREATE_QUERY = "INSERT INTO bot_pre_order (id, project_id, service_id, bot_big_order_id, bot_big_order_config_id, server_id, server_quantity, storage_id, storage_quantity, status, group_key, bot_campaign_group_id, name, reserve_rate) VALUES (:id, :projectId, :serviceId, :bigOrderId, :bigOrderConfigId, :serverId, :serverQuantity, :storageId, :storageQuantity, cast(:status as bot_pre_order_status), :groupKey, :campaignGroupId, :name, :reserveRate)";
    private static final String UPDATE_QUERY = "UPDATE bot_pre_order SET server_quantity = :serverQuantity, storage_quantity = :storageQuantity, status = cast(:status as bot_pre_order_status), reserve_rate = :reserveRate WHERE id = :id";
    private static final String GET_ID_BY_IDS_AND_STATUS = "SELECT id FROM bot_pre_order WHERE id IN (:id) AND status = cast(:status as bot_pre_order_status)";
    private static final String GET_BY_IDS_QUERY = "SELECT * FROM bot_pre_order WHERE id IN (:id)";
    private static final String DELETE_QUERY = "DELETE FROM bot_pre_order WHERE id = :id";
    private static final String TRUNCATE_QUERY = "TRUNCATE bot_pre_order CASCADE";
    private static final String EXISTS_BY_CAMPAIGN_GROUP_ID = "SELECT EXISTS(SELECT 1 FROM bot_pre_order WHERE bot_campaign_group_id = :campaign_group_id) AS exists";
    private static final String EXISTS_BY_CAMPAIGN_GROUP_ID_AND_ORDER_ID_NOT_IN = "SELECT EXISTS(SELECT 1 FROM bot_pre_order WHERE bot_campaign_group_id = :campaign_group_id AND bot_big_order_id NOT IN (:order_ids)) as exists";
    private static final String GET_SIMPLE_PRE_ORDERS = "SELECT bpo.id, bpo.service_id, " +
            "(bc.json ->> 0)::jsonb->>'fullName' configuration_name, bbo.date big_order_date, " +
            "(select e->>'locationKey' from jsonb_array_elements(bbo.configs) e where (e->>'id')::bigint = bpo.bot_big_order_config_id limit 1) big_order_location " +
            "FROM bot_pre_order bpo JOIN bot_configuration bc ON bc.id = bpo.server_id JOIN bot_big_order bbo ON bbo.id = bpo.bot_big_order_id " +
            "WHERE bpo.id in (:id)";

    private static final String GET_PRE_ORDERS_BY_BIG_ORDER_ID = "SELECT bpo.* FROM bot_pre_order bpo " +
            "WHERE bpo.bot_big_order_id in (:bigOrderId)";

    @Override
    public void lockForChanges() {
        acquireRowExclusiveLockOnTable("bot_pre_order");
    }

    @Override
    public Set<Long> getOrdersIdsInStatus(final Set<Long> preOrderIdsToCheck, final MappedPreOrder.Status status) {
        if (preOrderIdsToCheck.isEmpty()) {
            return Collections.emptySet();
        }
        return jdbcTemplate.queryForSet(GET_ID_BY_IDS_AND_STATUS, ImmutableMap.of(
                "id", preOrderIdsToCheck,
                "status", status.name()
        ), (rs, e) -> rs.getLong("id"));
    }

    @Override
    public boolean hasPreOrdersInCampaignGroup(final long campaignGroupId) {
        return jdbcTemplate.queryForObject(EXISTS_BY_CAMPAIGN_GROUP_ID, new MapSqlParameterSource("campaign_group_id", campaignGroupId), (rs, rn) -> {
            return rs.getBoolean("exists");
        });
    }

    @Override
    public boolean hasPreOrdersInCampaignGroupForOrdersOtherThan(final long campaignGroupId, final Set<Long> orderIds) {
        final MapSqlParameterSource params = new MapSqlParameterSource(ImmutableMap.of("campaign_group_id", campaignGroupId, "order_ids", orderIds));
        return jdbcTemplate.queryForObject(EXISTS_BY_CAMPAIGN_GROUP_ID_AND_ORDER_ID_NOT_IN, params, (rs, rn) -> {
            return rs.getBoolean("exists");
        });
    }

    @Override
    public void createAll(final Collection<MappedPreOrder> preOrders) {
        jdbcTemplate.batchUpdate(CREATE_QUERY, toParams(preOrders));
    }

    @Override
    public Optional<MappedPreOrder> readById(final long id) {
        return jdbcTemplate.queryForOptional(GET_BY_IDS_QUERY, Collections.singletonMap("id", id), this::toMappedPreOrder);
    }

    @Override
    public Set<MappedPreOrder> readByProjectAndServiceAndBigOrders(final Project project, final Set<Service> services,
                                                                   final Collection<Long> bigOrderIds, final long campaignGroupId) {
        if (bigOrderIds.isEmpty() || services.isEmpty()) {
            return Collections.emptySet();
        }
        return jdbcTemplate.queryForSet(GET_ORDERS_BY_PROJECT_AND_SERVICE_AND_BIG_ORDER_QUERY, ImmutableMap.of(
                "projectId", project.getId(),
                "bigOrderIds", new HashSet<>(bigOrderIds),
                "serviceId", ids(services),
                "campaignGroupId", campaignGroupId
        ), this::toMappedPreOrder);
    }


    @Override
    public boolean updateAll(final Collection<MappedPreOrder> preOrders) {
        if (preOrders.isEmpty()) {
            return false;
        }
        final int[] ints = jdbcTemplate.batchUpdate(UPDATE_QUERY, toParams(preOrders));
        return isSomeUpdated(ints);
    }

    @Override
    public boolean deleteAll(final Collection<MappedPreOrder> preOrders) {
        if (preOrders.isEmpty()) {
            return false;
        }
        final int[] ints = jdbcTemplate.batchUpdate(DELETE_QUERY, toParams(preOrders));
        return isSomeUpdated(ints);
    }

    @Override
    public boolean clear() {
        return jdbcTemplate.update(TRUNCATE_QUERY) > 0;
    }

    @Override
    public Collection<SimplePreOrder> readSimplePreOrders(final Set<Long> ids) {
        if (ids.isEmpty()) {
            return Collections.emptyList();
        }
        return jdbcTemplate.queryForSet(GET_SIMPLE_PRE_ORDERS, Collections.singletonMap("id", ids), this::toSimplePreOrder);
    }

    @Override
    public Map<Long, MappedPreOrder> getPreOrdersByBigOrderIds(final Set<Long> bigOrderIds) {
        if (bigOrderIds.isEmpty()) {
            return Collections.emptyMap();
        }
        final Map<Long, MappedPreOrder> result = new HashMap<>();
        jdbcTemplate.query(GET_PRE_ORDERS_BY_BIG_ORDER_ID, Collections.singletonMap("bigOrderId", bigOrderIds), (rs) -> {
            final MappedPreOrder preOrder = toMappedPreOrder(rs, -1);
            result.put(preOrder.getId(), preOrder);
        });
        return result;
    }

    private MappedPreOrder toMappedPreOrder(final ResultSet resultSet, final int i) throws SQLException {

        final Project project = Hierarchy.get().getProjectReader().read(resultSet.getLong("project_id"));
        final Service service = Hierarchy.get().getServiceReader().read(resultSet.getLong("service_id"));

        return new MappedPreOrder.Builder()
                .id(resultSet.getLong("id"))
                .project(project)
                .service(service)
                .bigOrderId(resultSet.getLong("bot_big_order_id"))
                .bigOrderConfigId(resultSet.getLong("bot_big_order_config_id"))
                .serverId(getLong(resultSet, "server_id"))
                .serverQuantity(getLong(resultSet, "server_quantity"))
                .storageId(getLong(resultSet, "storage_id"))
                .storageQuantity(getLong(resultSet, "storage_quantity"))
                .status(MappedPreOrder.Status.valueOf(resultSet.getString("status")))
                .groupKey(resultSet.getString("group_key"))
                .campaignGroupId(resultSet.getLong("bot_campaign_group_id"))
                .name(resultSet.getString("name"))
                .reserveRate(resultSet.getDouble("reserve_rate"))
                .build();
    }

    private SimplePreOrder toSimplePreOrder(final ResultSet resultSet, final int i) throws SQLException {
        final Service service = Hierarchy.get().getServiceReader().read(resultSet.getLong("service_id"));

        return new SimplePreOrder(
                resultSet.getLong("id"),
                service,
                resultSet.getString("configuration_name"),
                resultSet.getString("big_order_date"),
                resultSet.getString("big_order_location")
        );
    }

    private Map<String, ?> toParams(final MappedPreOrder order) {
        final HashMap<String, Object> map = new HashMap<>();
        map.put("id", order.getId());
        map.put("name", order.getName());
        map.put("projectId", order.getProject().getId());
        map.put("serviceId", order.getService().getId());
        map.put("bigOrderId", order.getBigOrderId());
        map.put("bigOrderConfigId", order.getBigOrderConfigId());
        map.put("serverId", order.getServerId());
        map.put("serverQuantity", order.getServerQuantity());
        map.put("storageId", order.getStorageId());
        map.put("storageQuantity", order.getStorageQuantity());
        map.put("status", order.getStatus().name());
        map.put("groupKey", order.getGroupKey());
        map.put("campaignGroupId", order.getCampaignGroupId());
        map.put("reserveRate", order.getReserveRate());
        return map;
    }

    private List<Map<String, ?>> toParams(final Collection<MappedPreOrder> orders) {
        return orders.stream()
                .map(this::toParams)
                .collect(Collectors.toList());
    }

    private boolean isSomeUpdated(final int[] ints) {
        for (final int updatedCount : ints) {
            if (updatedCount > 0) {
                return true;
            }
        }
        return false;
    }

}
