package ru.yandex.qe.dispenser.domain.dao.bot.preorder.change;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.inject.Inject;

import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.HashBasedTable;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Table;
import org.springframework.context.annotation.Lazy;

import ru.yandex.qe.dispenser.domain.QuotaChangeRequest;
import ru.yandex.qe.dispenser.domain.dao.bot.preorder.MappedPreOrderDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestDaoImpl;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;
import ru.yandex.qe.dispenser.domain.index.LongIndexable;

public class BotPreOrderChangeDaoImpl implements BotPreOrderChangeDao {

    private final QuotaChangeRequestDaoImpl quotaChangeRequestDao;
    private final MappedPreOrderDaoImpl preOrderDao;

    @Inject
    public BotPreOrderChangeDaoImpl(@Lazy final QuotaChangeRequestDaoImpl quotaChangeRequestDao,
                                    @Lazy final MappedPreOrderDaoImpl preOrderDao) {
        this.quotaChangeRequestDao = quotaChangeRequestDao;
        this.preOrderDao = preOrderDao;
    }

    protected final Map<Long, Map<Long, Double>> changeIdsByPreOrderId = new HashMap<>();

    @Override
    public void updateOrderChanges(final long id, final Map<Long, Double> changeIds) {
        changeIdsByPreOrderId.put(id, changeIds);
    }

    @Override
    public void setOrderChanges(final long id, final Map<Long, Double> changeIds) {
        if (changeIdsByPreOrderId.containsKey(id)) {
            throw new IllegalArgumentException("Changes for preOrder '" + id + "' already exists!");
        }
        changeIdsByPreOrderId.put(id, changeIds);
    }

    @Override
    public Set<Long> getOrderChangeIds(final long preOrderId) {
        return changeIdsByPreOrderId.get(preOrderId).keySet();
    }

    @Override
    public Multimap<Long, Long> getRequestsIdsByPreOrderIds(final Collection<Long> preOrderIds) {
        final HashMultimap<Long, Long> result = HashMultimap.create();

        final Set<Long> changeIds = new HashSet<>();

        final ArrayListMultimap<Long, Long> preOrderIdByChangeId = ArrayListMultimap.create();

        for (final Long preOrderId : preOrderIds) {
            final Set<Long> preOrderChangeIds = changeIdsByPreOrderId.get(preOrderId).keySet();
            changeIds.addAll(preOrderChangeIds);
            for (final Long changeId : preOrderChangeIds) {
                preOrderIdByChangeId.put(changeId, preOrderId);
            }
        }

        for (final QuotaChangeRequest quotaChangeRequest : quotaChangeRequestDao.getAll()) {

            quotaChangeRequest.getChanges().stream()
                    .map(LongIndexable::getId)
                    .filter(changeIds::contains)
                    .flatMap(id -> preOrderIdByChangeId.get(id).stream())
                    .forEach(orderId -> result.put(orderId, quotaChangeRequest.getId()));
        }

        return result;
    }

    @Override
    public boolean clear() {
        final int size = changeIdsByPreOrderId.size();
        changeIdsByPreOrderId.clear();
        return size > 0;
    }

    @Override
    public Table<Long, Long, Double> getPreOrderIdsByChangeId(final Collection<Long> requestIds) {
        final Collection<QuotaChangeRequest> requests = quotaChangeRequestDao.read(requestIds).values();

        final Set<Long> changeIds = requests.stream()
                .flatMap(r -> r.getChanges().stream())
                .map(QuotaChangeRequest.Change::getId)
                .collect(Collectors.toSet());

        final Table<Long, Long, Double> result = HashBasedTable.create();

        for (final Long orderId : changeIdsByPreOrderId.keySet()) {
            final Set<Long> preOrderChanges = changeIdsByPreOrderId.get(orderId).keySet();
            for (final Long changeId : preOrderChanges) {
                if (changeIds.contains(changeId)) {
                    result.put(changeId, orderId, changeIdsByPreOrderId.get(orderId).get(changeId));
                }
            }
        }

        return result;
    }

    @Override
    public Table<Long, Long, Double> getPreOrderChangesByBigOrderId(final Collection<Long> bigOrderIds) {
        final Set<Long> preOrderIds = preOrderDao.filter(po -> bigOrderIds.contains(po.getBigOrderId()))
                .map(LongIndexBase::getId)
                .collect(Collectors.toSet());

        final Table<Long, Long, Double> result = HashBasedTable.create();

        for (final Long orderId : changeIdsByPreOrderId.keySet()) {
            if (preOrderIds.contains(orderId)) {
                final Set<Long> preOrderChanges = changeIdsByPreOrderId.get(orderId).keySet();
                for (final Long changeId : preOrderChanges) {
                    result.put(changeId, orderId, changeIdsByPreOrderId.get(orderId).get(changeId));
                }
            }
        }

        return result;
    }

    @Override
    public Multimap<Long, Long> getPreOrderIdsByRequestId(final Collection<Long> requestIds) {
        final HashMultimap<Long, Long> result = HashMultimap.create();

        final ArrayListMultimap<Long, Long> preOrderIdsByChangeIds = ArrayListMultimap.create();
        for (final Long orderId : changeIdsByPreOrderId.keySet()) {
            for (final Long changeId : changeIdsByPreOrderId.get(orderId).keySet()) {
                preOrderIdsByChangeIds.put(changeId, orderId);
            }
        }

        for (final QuotaChangeRequest request : quotaChangeRequestDao.read(requestIds).values()) {
            for (final QuotaChangeRequest.Change change : request.getChanges()) {
                result.putAll(request.getId(), preOrderIdsByChangeIds.get(change.getId()));
            }
        }
        return result;
    }
}
