package ru.yandex.qe.dispenser.domain.dao.bot.preorder.change;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.HashBasedTable;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Multimap;
import com.google.common.collect.Table;

import ru.yandex.qe.dispenser.domain.dao.SqlDaoBase;

public class SqlBotPreOrderChangeDao extends SqlDaoBase implements BotPreOrderChangeDao {

    private static final String SET_ORDER_CHANGES = "INSERT INTO bot_pre_order_change (quota_request_change_id, bot_pre_order_id, proportion) VALUES (:changeId, :preOrderId, :proportion)";
    private static final String REMOVE_ORDER_CHANGES = "DELETE FROM bot_pre_order_change WHERE bot_pre_order_id in (:preOrderId)";
    private static final String GET_ORDER_CHANGES = "SELECT * FROM bot_pre_order_change WHERE bot_pre_order_id in (:preOrderId)";

    private static final String GET_PRE_ORDER_IDS_BY_REQUEST_ID = "SELECT * FROM bot_pre_order_change poc JOIN quota_request_change qrc ON qrc.id = poc.quota_request_change_id WHERE qrc.quota_request_id IN (:quotaRequestId)";
    private static final String GET_REQUEST_IDS_BY_PER_ORDER_ID = "SELECT * FROM bot_pre_order_change poc JOIN quota_request_change qrc ON qrc.id = poc.quota_request_change_id WHERE poc.bot_pre_order_id IN (:preOrderId)";

    private static final String GET_ORDER_CHANGES_BY_BIG_ORDER_ID = "SELECT bpoc.* FROM bot_pre_order_change bpoc " +
            "JOIN bot_pre_order bpo ON bpo.id = bpoc.bot_pre_order_id " +
            "WHERE bpo.bot_big_order_id in (:bigOrderId)";

    private static final String TRUNCATE_QUERY = "TRUNCATE bot_pre_order_change CASCADE";

    @Override
    public void updateOrderChanges(final long id, final Map<Long, Double> changeIds) {
        jdbcTemplate.update(REMOVE_ORDER_CHANGES, Collections.singletonMap("preOrderId", id));
        setOrderChanges(id, changeIds);
    }

    @Override
    public void setOrderChanges(final long id, final Map<Long, Double> changeIds) {
        if (changeIds.isEmpty()) {
            return;
        }
        final List<Map<String, ?>> params = changeIds.entrySet().stream()
                .map(e -> ImmutableMap.of(
                        "changeId", e.getKey(),
                        "preOrderId", id,
                        "proportion", e.getValue()
                ))
                .collect(Collectors.toList());

        jdbcTemplate.batchUpdate(SET_ORDER_CHANGES, params);
    }

    @Override
    public Set<Long> getOrderChangeIds(final long preOrderId) {
        return jdbcTemplate.queryForSet(GET_ORDER_CHANGES, Collections.singletonMap("preOrderId", preOrderId), (r, e) -> r.getLong(("quota_request_change_id")));
    }

    @Override
    public Multimap<Long, Long> getRequestsIdsByPreOrderIds(final Collection<Long> preOrderIds) {
        final HashMultimap<Long, Long> result = HashMultimap.create();
        if (preOrderIds.isEmpty()) {
            return result;
        }
        jdbcTemplate.query(GET_REQUEST_IDS_BY_PER_ORDER_ID, Collections.singletonMap("preOrderId", preOrderIds),
                (rs) -> result.put(rs.getLong("bot_pre_order_id"), rs.getLong("quota_request_id"))
        );
        return result;
    }

    @Override
    public boolean clear() {
        return jdbcTemplate.update(TRUNCATE_QUERY) > 0;
    }

    @Override
    public Multimap<Long, Long> getPreOrderIdsByRequestId(final Collection<Long> requestIds) {
        final HashMultimap<Long, Long> result = HashMultimap.create();
        if (requestIds.isEmpty()) {
            return result;
        }
        jdbcTemplate.query(GET_PRE_ORDER_IDS_BY_REQUEST_ID, Collections.singletonMap("quotaRequestId", requestIds),
                (rs) -> result.put(rs.getLong("quota_request_id"), rs.getLong("bot_pre_order_id"))
        );
        return result;
    }

    @Override
    public Table<Long, Long, Double> getPreOrderIdsByChangeId(final Collection<Long> requestIds) {
        final Table<Long, Long, Double> result = HashBasedTable.create();
        if (requestIds.isEmpty()) {
            return result;
        }
        jdbcTemplate.query(GET_PRE_ORDER_IDS_BY_REQUEST_ID, Collections.singletonMap("quotaRequestId", requestIds),
                rs -> result.put(rs.getLong("quota_request_change_id"), rs.getLong("bot_pre_order_id"), rs.getDouble("proportion"))
        );
        return result;
    }

    @Override
    public Table<Long, Long, Double> getPreOrderChangesByBigOrderId(final Collection<Long> bigOrderIds) {
        final Table<Long, Long, Double> result = HashBasedTable.create();
        if (bigOrderIds.isEmpty()) {
            return result;
        }
        jdbcTemplate.query(GET_ORDER_CHANGES_BY_BIG_ORDER_ID, Collections.singletonMap("bigOrderId", bigOrderIds),
                rs -> result.put(rs.getLong("quota_request_change_id"), rs.getLong("bot_pre_order_id"), rs.getDouble("proportion"))
        );
        return result;
    }
}
