package ru.yandex.qe.dispenser.domain.dao.bot.service;

import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.bot.BotServiceConfiguration;
import ru.yandex.qe.dispenser.domain.bot.Configuration;
import ru.yandex.qe.dispenser.domain.bot.ConfigurationWithComponents;
import ru.yandex.qe.dispenser.domain.dao.KeyExtractorInMemoryDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.bot.BotConfiguration;
import ru.yandex.qe.dispenser.domain.dao.bot.configuration.BotConfigurationDaoImpl;

import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

public class BotServiceConfigurationDaoImpl extends KeyExtractorInMemoryDaoImpl<BotServiceConfiguration, BotServiceConfiguration>
        implements BotServiceConfigurationDao {
    @Autowired
    private BotConfigurationDaoImpl configurationDao;

    public BotServiceConfigurationDaoImpl() {
        super(Function.identity());
    }

    @Override
    public @NotNull Stream<BotServiceConfiguration> filter(final @NotNull Predicate<BotServiceConfiguration> predicate) {
        final Set<Long> allIds = configurationDao.getAllIds();
        return super.filter(serviceConfiguration -> allIds.contains(serviceConfiguration.getConfigurationId()))
                .filter(predicate);
    }

    @Override
    public Set<BotServiceConfiguration> getByService(final Service service) {
        return getByServices(Collections.singleton(service));
    }

    @Override
    public Set<BotServiceConfiguration> getByService(final Service service,
                                                     final long botCampaignGroupId) {
        return getByServices(Collections.singleton(service), botCampaignGroupId);
    }

    @Override
    public Set<BotServiceConfiguration> getByServices(final Collection<Service> services,
                                                      final long botCampaignGroupId) {
        return filter(botServiceConfiguration -> services.contains(botServiceConfiguration.getService())
                && botServiceConfiguration.getBotCampaignGroupId() == botCampaignGroupId)
                .collect(Collectors.toSet());
    }

    @Override
    public Set<BotServiceConfiguration> getByServices(final Collection<Service> services) {
        return filter(botServiceConfiguration -> services.contains(botServiceConfiguration.getService()))
                .collect(Collectors.toSet());
    }

    @Override
    public Set<BotServiceConfiguration.View> getByServiceViews(Service service) {
        final Set<BotServiceConfiguration> byServices = getByServices(Collections.singleton(service));
        return getByServiceViews(byServices);
    }

    @Override
    public Set<BotServiceConfiguration.View> getByServiceViews(final Service service,
                                                               final long botCampaignGroupId) {
        final Set<BotServiceConfiguration> serviceConfigurations = getByService(service, botCampaignGroupId);
        return getByServiceViews(serviceConfigurations);
    }

    private Set<BotServiceConfiguration.View> getByServiceViews(Set<BotServiceConfiguration> serviceConfigurations) {
        final Set<Long> ids = serviceConfigurations.stream()
                .map(BotServiceConfiguration::getConfigurationId)
                .collect(Collectors.toSet());

        final Map<Long, Configuration> configurationsById = configurationDao.getByIds(ids).stream()
                .map(BotConfiguration::getConfiguration)
                .collect(Collectors.toMap(Configuration::getId, Function.identity()));

        return serviceConfigurations.stream()
                .map(serviceConfiguration -> {

                    final Configuration configuration = configurationsById.get(serviceConfiguration.getConfigurationId());

                    if (configuration instanceof ConfigurationWithComponents) {
                        return serviceConfiguration.toView((ConfigurationWithComponents) configuration);
                    }
                    throw new IllegalStateException("Configuration without components");
                })
                .collect(Collectors.toSet());
    }

    @Override
    public void deleteByService(final Service service, final long botCampaignGroupId) {
        deleteAll(getByServices(Collections.singleton(service), botCampaignGroupId));
    }

    @Override
    public void createAll(final Set<BotServiceConfiguration> configurations) {
        configurations.forEach(this::create);
    }

    @Override
    public Set<Long> getConfigurationsIds() {
        return getAll().stream()
                .map(BotServiceConfiguration::getConfigurationId)
                .collect(Collectors.toSet());
    }

    @Override
    public void updateAll(final Set<BotServiceConfiguration> configurations) {
        super.updateAll(configurations);
    }
}
