package ru.yandex.qe.dispenser.domain.dao.bot.service.reserve;

import java.io.Serializable;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.builder.CompareToBuilder;
import org.jetbrains.annotations.NotNull;

import ru.yandex.qe.dispenser.api.Keyable;
import ru.yandex.qe.dispenser.api.v1.DiAmount;
import ru.yandex.qe.dispenser.api.v1.DiBotServiceReserve;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;

@ParametersAreNonnullByDefault
public class BotServiceReserve extends LongIndexBase implements Keyable<BotServiceReserve.Key>, BotReserve {
    private final Key key;
    private final Long amount;

    public BotServiceReserve(final Key key, final Long amount) {
        this.key = key;
        this.amount = amount;
    }

    public Resource getResource() {
        return key.resource;
    }

    @Override
    public Service getService() {
        return getResource().getService();
    }

    public Set<Segment> getSegments() {
        return key.segments;
    }

    public Long getAmount() {
        return amount;
    }

    public Long getBigOrderId() {
        return key.bigOrderId;
    }

    public Long getCampaignId() {
        return key.campaignId;
    }

    public DiBotServiceReserve toView() {
        return DiBotServiceReserve.forResource(getResource().getPublicKey())
                .withAmount(DiAmount.of(getAmount(), getResource().getType().getBaseUnit()))
                .withBigOrder(getBigOrderId())
                .withSegments(getSegments().stream().map(Segment::getPublicKey).collect(Collectors.toSet()))
                .forService(getService().getKey())
                .withCampaignId(getCampaignId())
                .withId(getId())
                .build();
    }

    @NotNull
    @Override
    public Key getKey() {
        return key;
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        final BotServiceReserve that = (BotServiceReserve) o;
        return Objects.equals(key, that.key) &&
                Objects.equals(getId(), that.getId()) &&
                Objects.equals(amount, that.amount);
    }

    @Override
    public int hashCode() {
        return Objects.hash(key, amount, getId());
    }

    @Override
    public String toString() {
        return "BotServiceReserve{" +
                "id=" + getId() +
                ", key=" + key +
                ", amount=" + amount +
                '}';
    }

    public static class Key implements Comparable<Key>, Serializable {
        private final Resource resource;
        private final Set<Segment> segments;
        private final Long bigOrderId;
        private final Long campaignId;

        public Key(final Resource resource, final Set<Segment> segments, final Long bigOrderId, final Long campaignId) {
            this.resource = resource;
            this.segments = segments;
            this.bigOrderId = bigOrderId;
            this.campaignId = campaignId;
        }

        public Resource getResource() {
            return resource;
        }

        public Set<Segment> getSegments() {
            return segments;
        }

        public Long getBigOrderId() {
            return bigOrderId;
        }

        public Long getCampaignId() {
            return campaignId;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            final Key key = (Key) o;
            return resource.equals(key.resource) &&
                    segments.equals(key.segments) &&
                    bigOrderId.equals(key.bigOrderId) &&
                    campaignId.equals(key.campaignId);
        }

        @Override
        public int hashCode() {
            return Objects.hash(resource, segments, bigOrderId, campaignId);
        }

        @Override
        public int compareTo(@NotNull final BotServiceReserve.Key o) {
            return new CompareToBuilder()
                    .append(resource, o.resource)
                    .append(segments, o.segments)
                    .append(campaignId, o.campaignId)
                    .append(bigOrderId, o.bigOrderId)
                    .build();
        }

        @Override
        public String toString() {
            return "Key{" +
                    "resource=" + resource +
                    ", segments=" + segments +
                    ", bigOrderId=" + bigOrderId +
                    ", campaignId=" + campaignId +
                    '}';
        }
    }
}
