package ru.yandex.qe.dispenser.domain.dao.bot.service.reserve.configurations;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.jetbrains.annotations.NotNull;
import org.springframework.dao.DuplicateKeyException;

import ru.yandex.qe.dispenser.domain.dao.InMemoryLongKeyDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.bot.service.reserve.BotServiceReserveFilter;

public class BotConfigurationReserveDaoImpl extends InMemoryLongKeyDaoImpl<BotConfigurationReserve> implements BotConfigurationReserveDao {

    private final Map<BotConfigurationReserve.Key, Long> keyToId = new HashMap<>();

    @Override
    public @NotNull BotConfigurationReserve create(final @NotNull BotConfigurationReserve newInstance) {
        final BotConfigurationReserve.Key key = newInstance.getKey();
        if (keyToId.containsKey(key)) {
            throw new DuplicateKeyException(newInstance.getClass().getSimpleName() + " object with key " + key + " already exists!");
        }

        final BotConfigurationReserve reserve = super.create(newInstance);
        keyToId.put(key, reserve.getId());
        return reserve;
    }

    @Override
    public boolean update(final @NotNull BotConfigurationReserve obj) {
        final BotConfigurationReserve.Key key = obj.getKey();
        final BotConfigurationReserve.Key previousKey = read(obj.getId()).getKey();
        if (!key.equals(previousKey)) {
            if (keyToId.containsKey(key)) {
                throw new DuplicateKeyException(obj.getClass().getSimpleName() + " object with key " + key + " already exists!");
            }
        }
        final boolean update = super.update(obj);
        keyToId.remove(previousKey);
        keyToId.put(key, obj.getId());
        return update;
    }

    @Override
    public boolean delete(final @NotNull BotConfigurationReserve obj) {
        final boolean delete = super.delete(obj);
        keyToId.remove(obj.getKey());
        return delete;
    }

    @Override
    public Collection<BotConfigurationReserve> getReserves(final BotServiceReserveFilter filter) {
        return getReserveStreamFiltered(filter).collect(Collectors.toList());
    }

    @Override
    public Stream<BotConfigurationReserve> getReserveStreamFiltered(final BotServiceReserveFilter filter) {
        final List<Predicate<BotConfigurationReserve>> predicates = new ArrayList<>();
        if (!filter.getServices().isEmpty()) {
            predicates.add(reserve -> filter.getServices().contains(reserve.getKey().getService()));
        }

        if (!filter.getCampaignIds().isEmpty()) {
            predicates.add(reserve -> filter.getCampaignIds().contains(reserve.getKey().getCampaignId()));
        }

        if (!filter.getBigOrderIds().isEmpty()) {
            predicates.add(reserve -> filter.getBigOrderIds().contains(reserve.getKey().getBigOrderId()));
        }

        if (!filter.getCampaignBigOrders().isEmpty()) {
            predicates.add(reserve -> filter.getCampaignBigOrders().stream()
                    .anyMatch(co -> Long.valueOf(co.getBigOrderId()).equals(reserve.getKey().getBigOrderId())
                            && Long.valueOf(co.getCampaignId()).equals(reserve.getKey().getCampaignId())));
        }

        return predicates.stream()
                .reduce(Predicate::and)
                .map(this::filter)
                .orElseGet(() -> getAll().stream());
    }

    @Override
    public boolean clear() {
        keyToId.clear();
        return super.clear();
    }
}
