package ru.yandex.qe.dispenser.domain.dao.bot.settings;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Comparator;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import javax.inject.Inject;

import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.dao.EmptyResultDataAccessException;
import ru.yandex.qe.dispenser.domain.BotCampaignGroup;
import ru.yandex.qe.dispenser.domain.BotCampaignGroupUpdate;
import ru.yandex.qe.dispenser.domain.CampaignForBot;
import ru.yandex.qe.dispenser.domain.bot.BigOrder;
import ru.yandex.qe.dispenser.domain.dao.InMemoryLongKeyDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.bot.bigorder.BigOrderDao;
import ru.yandex.qe.dispenser.domain.dao.campaign.CampaignDao;
import ru.yandex.qe.dispenser.domain.util.CollectionUtils;

public class BotCampaignGroupDaoImpl extends InMemoryLongKeyDaoImpl<BotCampaignGroup> implements BotCampaignGroupDao {

    @Inject
    private BigOrderDao bigOrderDao;

    @Inject
    private CampaignDao campaignDao;

    @Override
    public @NotNull BotCampaignGroup create(final @NotNull BotCampaignGroup newInstance) {
        checkPreOrdersUniqueness(null, newInstance.getBigOrders());
        checkCampaignsUniqueness(null, newInstance.getCampaigns());
        return super.create(newInstance);
    }

    private void checkPreOrdersUniqueness(@Nullable final Long groupId, final Collection<BigOrder> simpleBigOrders) {
        checkPreOrdersUniqueness(groupId, CollectionUtils.ids(simpleBigOrders));
    }

    private void checkPreOrdersUniqueness(@Nullable final Long groupId, final Set<Long> preOrders) {
        filter(g -> !Objects.equals(g.getId(), groupId)).forEach(g -> {
            final Set<Long> groupOrders = CollectionUtils.ids(g.getBigOrders());
            if (!Sets.intersection(groupOrders, preOrders).isEmpty()) {
                throw new IllegalArgumentException("Big orders cannot be used in several campaign groups");
            }
        });
    }

    private void checkCampaignsUniqueness(@Nullable final Long groupId, final Collection<CampaignForBot> campaigns) {
        checkCampaignsUniqueness(groupId, CollectionUtils.ids(campaigns));
    }

    private void checkCampaignsUniqueness(@Nullable final Long groupId, final Set<Long> campaigns) {
        filter(g -> Objects.equals(g.getId(), groupId)).forEach(c -> {
            final Set<Long> groupCampaigns = CollectionUtils.ids(c.getCampaigns());
            if (!Sets.intersection(groupCampaigns, campaigns).isEmpty()) {
                throw new IllegalArgumentException("Campaigns cannot be used in several campaign groups");
            }
        });
    }

    @Override
    public @NotNull BotCampaignGroup read(final @NotNull Long id) {
        final BotCampaignGroup readed = super.read(id);
        final Map<Long, CampaignForBot> campaigns = campaignDao.readForBot(CollectionUtils.ids(readed.getCampaigns()));
        return readed.copyBuilder().setCampaigns(new ArrayList<>(campaigns.values())).build();
    }

    @Override
    public Set<BotCampaignGroup> getActiveGroups() {
        return filter(BotCampaignGroup::isActive)
                .collect(Collectors.toSet());
    }

    @Override
    public Optional<BotCampaignGroup> readOptional(long id) {
        final Optional<BotCampaignGroup> readed = Optional.ofNullable(id2obj.get(id));
        return readed.map(botCampaignGroup -> {
            final Map<Long, CampaignForBot> campaigns = campaignDao.readForBot(CollectionUtils.ids(botCampaignGroup.getCampaigns()));
            return botCampaignGroup.copyBuilder().setCampaigns(new ArrayList<>(campaigns.values())).build();
        });
    }

    @Override
    public BotCampaignGroup readForUpdate(final long id) throws EmptyResultDataAccessException {
        return read(id);
    }

    @Override
    public Optional<BotCampaignGroup> getByCampaign(final long campaignId) {
        return filter(g -> g.getCampaigns().stream().anyMatch(cp -> cp.getId() == campaignId))
                .findFirst();
    }

    @Override
    public boolean hasGroupForCampaign(final long campaignId) {
        return getByCampaign(campaignId).isPresent();
    }

    @Override
    public BotCampaignGroup partialUpdate(final BotCampaignGroup group, final BotCampaignGroupUpdate update) {
        final BotCampaignGroup.Builder builder = group.copyBuilder();
        if (update.getKey() != null) {
            builder.setKey(update.getKey());
        }
        if (update.getName() != null) {
            builder.setName(update.getName());
        }
        if (update.getActive() != null) {
            builder.setActive(update.getActive());
        }
        if (update.getBigOrders() != null) {
            final Set<BigOrder> orders = bigOrderDao.getByIds(update.getBigOrders());
            checkPreOrdersUniqueness(group.getId(), orders);
            builder.setBigOrders(orders.stream().sorted(Comparator.comparing(BigOrder::getId)).collect(Collectors.toList()));
        }
        if (update.getBotPreOrderIssueKey() != null) {
            builder.setBotPreOrderIssueKey(update.getBotPreOrderIssueKey());
        }
        if (update.getCampaigns() != null) {
            final Map<Long, CampaignForBot> campaignById = campaignDao.readForBot(update.getCampaigns());
            checkCampaignsUniqueness(group.getId(), campaignById.values());
            builder.setCampaigns(campaignById.values().stream().sorted(Comparator.comparing(CampaignForBot::getId)).collect(Collectors.toList()));
        }
        final BotCampaignGroup updated = builder.build();
        update(updated);
        return updated;
    }

    @Override
    public void attachCampaigns(final long groupId, final Collection<Long> campaigns) {
        final BotCampaignGroup group = read(groupId);
        final Set<Long> allCampaignIds = Sets.union(CollectionUtils.ids(group.getCampaigns()), ImmutableSet.copyOf(campaigns));
        final Map<Long, CampaignForBot> toAttach = campaignDao.readForBot(allCampaignIds);
        final BotCampaignGroup updated = group.copyBuilder()
                .setCampaigns(toAttach.values().stream().sorted(Comparator.comparing(CampaignForBot::getStartDate)).collect(Collectors.toList()))
                .build();
        update(updated);
    }

    @Override
    public void detachCampaigns(final long groupId, final Collection<Long> campaigns) {
        final BotCampaignGroup group = read(groupId);
        final Set<Long> allCampaignIds = Sets.difference(CollectionUtils.ids(group.getCampaigns()), ImmutableSet.copyOf(campaigns));
        final Map<Long, CampaignForBot> toAttach = campaignDao.readForBot(allCampaignIds);
        final BotCampaignGroup updated = group.copyBuilder()
                .setCampaigns(toAttach.values().stream().sorted(Comparator.comparing(CampaignForBot::getStartDate)).collect(Collectors.toList()))
                .build();
        update(updated);
    }

    @Override
    public void attachBigOrders(final long groupId, final Collection<Long> bigOrders) {
        final BotCampaignGroup group = read(groupId);
        final Set<Long> allOrdersIds = Sets.union(CollectionUtils.ids(group.getBigOrders()), ImmutableSet.copyOf(bigOrders));
        final Set<BigOrder> orders = bigOrderDao.getByIds(allOrdersIds);
        final BotCampaignGroup updated = group.copyBuilder()
                .setBigOrders(orders.stream().sorted(Comparator.comparing(BigOrder::getId)).collect(Collectors.toList()))
                .build();
        update(updated);
    }

    @Override
    public void detachBigOrders(final long groupId, final Collection<Long> bigOrders) {
        final BotCampaignGroup group = read(groupId);
        final Set<Long> allOrdersIds = Sets.difference(CollectionUtils.ids(group.getBigOrders()), ImmutableSet.copyOf(bigOrders));
        final Set<BigOrder> orders = bigOrderDao.getByIds(allOrdersIds);
        final BotCampaignGroup updated = group.copyBuilder()
                .setBigOrders(orders.stream().sorted(Comparator.comparing(BigOrder::getId)).collect(Collectors.toList()))
                .build();
        update(updated);
    }
}
