package ru.yandex.qe.dispenser.domain.dao.campaign;

import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import javax.inject.Inject;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.domain.Campaign;
import ru.yandex.qe.dispenser.domain.dao.base_resources.CacheRefreshScheduler;

@Component
public class CampaignCache {

    private final CampaignDao campaignDao;
    private final LoadingCache<Long, Optional<Campaign>> cache;

    @Inject
    public CampaignCache(CampaignDao campaignDao) {
        this.campaignDao = campaignDao;
        this.cache = CacheBuilder.newBuilder()
                .expireAfterWrite(5, TimeUnit.MINUTES)
                .expireAfterAccess(24, TimeUnit.HOURS)
                .maximumSize(10000)
                .build(new Loader(campaignDao));
    }

    public Optional<Campaign> getById(long id) {
        return CacheRefreshScheduler.wrapException(() -> cache.get(id));
    }

    public Set<Campaign> getByIds(Collection<? extends Long> ids) {
        return CacheRefreshScheduler.wrapException(() -> cache.getAll(ids).values()
                .stream().flatMap(Optional::stream).collect(Collectors.toSet()));
    }

    public void clear() {
        cache.invalidateAll();
    }

    @Transactional(propagation = Propagation.REQUIRED)
    public void refresh() {
        List<Long> keysToRefresh = cache.asMap().keySet().stream().distinct().collect(Collectors.toList());
        Map<Long, Campaign> refreshed = Lists.partition(keysToRefresh, 500).stream()
                .flatMap(keys -> campaignDao.readByIds(new HashSet<>(keys)).stream())
                .collect(Collectors.toMap(Campaign::getId, Function.identity()));
        Map<Long, Optional<Campaign>> refreshedById = new HashMap<>();
        keysToRefresh.forEach(key -> refreshedById.put(key, Optional.ofNullable(refreshed.get(key))));
        cache.putAll(refreshedById);
    }

    private static final class Loader extends CacheLoader<Long, Optional<Campaign>> {

        private final CampaignDao campaignDao;

        private Loader(CampaignDao campaignDao) {
            this.campaignDao = campaignDao;
        }

        @Override
        public Optional<Campaign> load(Long key) {
            return campaignDao.readOptional(key);
        }

        @Override
        public Map<Long, Optional<Campaign>> loadAll(Iterable<? extends Long> keys) {
            List<Long> keySet = StreamSupport.stream(keys.spliterator(), false).distinct()
                    .collect(Collectors.toList());
            Map<Long, Campaign> knownEntitiesByKey = Lists.partition(keySet, 500).stream()
                    .flatMap(page -> campaignDao.readByIds(new HashSet<>(page)).stream())
                    .collect(Collectors.toMap(Campaign::getId, Function.identity()));
            Map<Long, Optional<Campaign>> result = new HashMap<>();
            keySet.forEach(key -> result.put(key, Optional.ofNullable(knownEntitiesByKey.get(key))));
            return result;
        }

    }

}
