package ru.yandex.qe.dispenser.domain.dao.campaign;

import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

import javax.inject.Inject;

import org.apache.commons.lang3.tuple.Pair;
import org.jetbrains.annotations.NotNull;

import ru.yandex.qe.dispenser.domain.CampaignResource;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segmentation;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.InMemoryLongKeyDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.resource.ResourceDao;
import ru.yandex.qe.dispenser.domain.dao.resource.segmentation.ResourceSegmentationDao;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;

public class CampaignResourceDaoImpl extends InMemoryLongKeyDaoImpl<CampaignResource> implements CampaignResourceDao {

    @NotNull
    private final ResourceDao resourceDao;
    @NotNull
    private final ResourceSegmentationDao resourceSegmentationDao;
    private final Map<Pair<Long, Long>, Long> campaignResourceUniqueMap = new ConcurrentHashMap<>();

    @Inject
    public CampaignResourceDaoImpl(@NotNull final ResourceDao resourceDao,
                                   @NotNull final ResourceSegmentationDao resourceSegmentationDao) {
        this.resourceDao = resourceDao;
        this.resourceSegmentationDao = resourceSegmentationDao;
    }

    @NotNull
    @Override
    public List<CampaignResource> getByCampaignId(final long campaignId) {
        return filter(r -> r.getCampaignId() == campaignId)
                .sorted(Comparator.comparing(LongIndexBase::getId)).collect(Collectors.toList());
    }

    @NotNull
    @Override
    public List<CampaignResource> getByResourceId(final long resourceId) {
        return filter(r -> r.getResourceId() == resourceId)
                .sorted(Comparator.comparing(LongIndexBase::getId)).collect(Collectors.toList());
    }

    @NotNull
    @Override
    public Map<Long, Set<Long>> getAvailableCampaignsForServices(@NotNull final Set<Service> services) {
        if (services.isEmpty()) {
            return Collections.emptyMap();
        }
        final Map<Long, Set<Resource>> resourcesByServiceId = services.stream()
                .collect(Collectors.toMap(LongIndexBase::getId, resourceDao::getByService));
        final Map<Long, Set<Long>> campaignsByResourceId = getAll().stream()
                .collect(Collectors.groupingBy(CampaignResource::getResourceId,
                        Collectors.mapping(CampaignResource::getCampaignId, Collectors.toSet())));
        final Map<Long, Set<Long>> result = new HashMap<>();
        services.forEach(s -> {
            final Set<Long> availableCampaigns = new HashSet<>();
            final Set<Resource> serviceResources = resourcesByServiceId.get(s.getId());
            if (serviceResources == null) {
                result.put(s.getId(), availableCampaigns);
                return;
            }
            serviceResources.forEach(r -> {
                final Set<Long> campaigns = campaignsByResourceId.get(r.getId());
                if (campaigns != null) {
                    availableCampaigns.addAll(campaigns);
                }
            });
            result.put(s.getId(), availableCampaigns);
        });
        return result;
    }

    @NotNull
    @Override
    public Map<Long, Set<Long>> getAvailableCampaignsForResources(@NotNull final Set<Resource> resources) {
        if (resources.isEmpty()) {
            return Collections.emptyMap();
        }
        final Map<Long, Set<Long>> campaignsByResourceId = getAll().stream()
                .collect(Collectors.groupingBy(CampaignResource::getResourceId,
                        Collectors.mapping(CampaignResource::getCampaignId, Collectors.toSet())));
        final Map<Long, Set<Long>> result = new HashMap<>();
        resources.forEach(r -> {
            result.put(r.getId(), campaignsByResourceId.getOrDefault(r.getId(), Collections.emptySet()));
        });
        return result;
    }

    @NotNull
    @Override
    public Map<Long, Set<Long>> getAvailableCampaignsForSegmentations(@NotNull final Set<Segmentation> segmentations) {
        if (segmentations.isEmpty()) {
            return Collections.emptyMap();
        }
        final Map<Long, Set<Long>> campaignsByResourceId = getAll().stream()
                .collect(Collectors.groupingBy(CampaignResource::getResourceId,
                        Collectors.mapping(CampaignResource::getCampaignId, Collectors.toSet())));
        final Map<Long, Set<Long>> result = new HashMap<>();
        segmentations.forEach(segmentation -> {
            final Set<Resource> resources = resourceSegmentationDao.getResources(segmentation);
            final Set<Long> segmentationCampaignsIds = new HashSet<>();
            resources.forEach(resource -> {
                final Set<Long> campaigns = campaignsByResourceId.getOrDefault(resource.getId(), Collections.emptySet());
                segmentationCampaignsIds.addAll(campaigns);
            });
            result.put(segmentation.getId(), segmentationCampaignsIds);
        });
        return result;
    }

    @Override
    public boolean existsByResource(@NotNull final Resource resource) {
        return getAll().stream().anyMatch(cr -> cr.getResourceId() == resource.getId());
    }

    @Override
    public boolean existsByService(@NotNull final Service service) {
        final Map<Long, Long> serviceIdByResourceId = resourceDao.getAll()
                .stream().collect(Collectors.toMap(LongIndexBase::getId, r -> r.getService().getId()));
        return getAll().stream()
                .anyMatch(cr -> Objects.equals(service.getId(), serviceIdByResourceId.get(cr.getResourceId())));
    }

    @NotNull
    @Override
    public CampaignResource create(final @NotNull CampaignResource newInstance) {
        final Pair<Long, Long> pair = Pair.of(newInstance.getCampaignId(), newInstance.getResourceId());
        if (campaignResourceUniqueMap.containsKey(pair)) {
            throw new IllegalArgumentException("Campaign resource with campaignId " + newInstance.getCampaignId() +
                    " and resourceId " + newInstance.getResourceId() + " already exist!");
        }
        final @NotNull CampaignResource result = super.create(newInstance);
        campaignResourceUniqueMap.put(pair, result.getId());
        return result;
    }

    @Override
    protected CampaignResource createUnsafe(final @NotNull Long id, final @NotNull CampaignResource newInstance) {
        final Pair<Long, Long> pair = Pair.of(newInstance.getCampaignId(), newInstance.getResourceId());
        if (campaignResourceUniqueMap.containsKey(pair)) {
            throw new IllegalArgumentException("Campaign resource with campaignId " + newInstance.getCampaignId() +
                    " and resourceId " + newInstance.getResourceId() + " already exist!");
        }
        final @NotNull CampaignResource result = super.createUnsafe(id, newInstance);
        campaignResourceUniqueMap.put(pair, result.getId());
        return result;
    }

    @NotNull
    @Override
    public CampaignResource createIfAbsent(final @NotNull CampaignResource obj) {
        final @NotNull CampaignResource result = super.createIfAbsent(obj);
        final Pair<Long, Long> pair = Pair.of(result.getCampaignId(), result.getResourceId());

        if (campaignResourceUniqueMap.containsKey(pair) && campaignResourceUniqueMap.get(pair) != result.getId()) {
            throw new IllegalArgumentException("Campaign resource with campaignId " + result.getCampaignId() +
                    " and resourceId " + result.getResourceId() + " already exist!");
        }
        campaignResourceUniqueMap.put(pair, result.getId());
        return result;
    }

    @Override
    public boolean update(final @NotNull CampaignResource obj) {
        final Pair<Long, Long> pair = Pair.of(obj.getCampaignId(), obj.getResourceId());
        if (campaignResourceUniqueMap.containsKey(pair) &&
                campaignResourceUniqueMap.get(pair) != obj.getId()) {
            throw new IllegalArgumentException("Campaign resource with campaignId " + obj.getCampaignId() +
                    " and resourceId " + obj.getResourceId() + " already exist!");
        }
        campaignResourceUniqueMap.entrySet().stream().filter(entry -> entry.getValue() == obj.getId())
                .collect(Collectors.toList()).forEach(entry -> campaignResourceUniqueMap.remove(entry.getKey()));
        campaignResourceUniqueMap.put(pair, obj.getId());
        return super.update(obj);
    }

    @Override
    public boolean delete(final @NotNull CampaignResource obj) {
        final Pair<Long, Long> pair = Pair.of(obj.getCampaignId(), obj.getResourceId());
        campaignResourceUniqueMap.remove(pair);
        return super.delete(obj);
    }
}
