package ru.yandex.qe.dispenser.domain.dao.campaign;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Sets;
import org.jetbrains.annotations.NotNull;
import org.postgresql.util.PGobject;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.domain.CampaignResource;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.Segmentation;
import ru.yandex.qe.dispenser.domain.Service;
import ru.yandex.qe.dispenser.domain.dao.SqlDaoBase;
import ru.yandex.qe.dispenser.domain.dao.SqlUtils;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;

public class SqlCampaignResourceDao extends SqlDaoBase implements CampaignResourceDao {

    private static final String COUNT_QUERY = "SELECT COUNT(*) FROM campaign_resource";
    private static final String EXISTS_QUERY = "SELECT COUNT(1) WHERE EXISTS (SELECT * FROM campaign_resource)";
    private static final String CLEAR_QUERY = "TRUNCATE campaign_resource CASCADE";
    private static final String EXISTS_BY_ID_QUERY = "SELECT COUNT(1) WHERE EXISTS (SELECT * FROM campaign_resource WHERE id = :id)";
    private static final String DELETE_BY_ID = "DELETE FROM campaign_resource WHERE id = :id";
    private static final String DELETE_BY_IDS = "DELETE FROM campaign_resource WHERE id IN (:ids)";
    private static final String INSERT_QUERY = "INSERT INTO campaign_resource (campaign_id, resource_id, required, \"default\", default_amount, settings) " +
            "VALUES (:campaign_id, :resource_id, :required, :default, :default_amount, :settings)";
    private static final String UPDATE_QUERY = "UPDATE campaign_resource SET campaign_id = :campaign_id, resource_id = :resource_id, " +
            "required = :required, \"default\" = :default, default_amount = :default_amount, settings = :settings WHERE id = :id";
    private static final String SELECT_ALL = "SELECT id, campaign_id, resource_id, required, \"default\", default_amount, settings FROM campaign_resource";
    private static final String SELECT_BY_ID = "SELECT id, campaign_id, resource_id, required, \"default\", default_amount, settings FROM " +
            "campaign_resource WHERE id = :id";
    private static final String SELECT_BY_IDS = "SELECT id, campaign_id, resource_id, required, \"default\", default_amount, settings FROM " +
            "campaign_resource WHERE id IN (:ids)";
    private static final String SELECT_BY_CAMPAIGN_ID = "SELECT id, campaign_id, resource_id, required, \"default\", default_amount, settings " +
            "FROM campaign_resource WHERE campaign_id = :campaign_id ORDER BY id ASC";
    private static final String SELECT_BY_RESOURCE_ID = "SELECT id, campaign_id, resource_id, required, \"default\", default_amount, settings " +
            "FROM campaign_resource WHERE resource_id = :resource_id ORDER BY id ASC";
    private static final String SELECT_CAMPAIGNS_BY_SERVICE_IDS = "SELECT cr.campaign_id, r.service_id FROM campaign_resource cr " +
            "JOIN resource r ON cr.resource_id = r.id WHERE r.service_id IN (:service_ids) GROUP BY r.service_id, cr.campaign_id " +
            "ORDER BY r.service_id ASC, cr.campaign_id ASC";
    private static final String SELECT_CAMPAIGNS_BY_RESOURCE_IDS = "SELECT campaign_id, resource_id FROM campaign_resource WHERE " +
            "resource_id IN (:resource_ids) ORDER BY resource_id ASC, campaign_id ASC";
    private static final String SELECT_CAMPAIGNS_BY_SEGMENTATION_IDS = "SELECT cr.campaign_id, rs.segmentation_id FROM campaign_resource cr " +
            "JOIN resource_segmentation rs ON cr.resource_id = rs.resource_id WHERE rs.segmentation_id IN (:segmentation_ids) " +
            "GROUP BY rs.segmentation_id, cr.campaign_id ORDER BY rs.segmentation_id ASC, cr.campaign_id ASC";
    private static final String EXISTS_BY_RESOURCE_ID_QUERY = "SELECT COUNT(1) WHERE EXISTS (SELECT * FROM campaign_resource WHERE resource_id = :resource_id)";
    private static final String EXISTS_BY_SERVICE_ID_QUERY = "SELECT COUNT(1) WHERE EXISTS " +
            "(SELECT * FROM campaign_resource cr JOIN resource r ON cr.resource_id = r.id WHERE r.service_id = :service_id)";

    @NotNull
    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Set<CampaignResource> getAll() {
        return jdbcTemplate.queryForSet(SELECT_ALL, Collections.emptyMap(), (rs, rowNum) -> mapRow(rs));
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public int size() {
        return jdbcTemplate.queryForObject(COUNT_QUERY, Collections.emptyMap(), Long.class).intValue();
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public boolean isEmpty() {
        return jdbcTemplate.queryForObject(EXISTS_QUERY, Collections.emptyMap(), Long.class) == 0;
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public boolean clear() {
        return jdbcTemplate.update(CLEAR_QUERY) > 0;
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public CampaignResource create(@NotNull final CampaignResource object) {
        final Map<String, Object> params = new HashMap<>();
        params.put("campaign_id", object.getCampaignId());
        params.put("resource_id", object.getResourceId());
        params.put("required", object.isRequired());
        params.put("default", object.isDefaultResource());
        params.put("default_amount", object.getDefaultAmount().orElse(null));
        params.put("settings", SqlUtils.toJsonb(object.getSettings()));
        final long id = jdbcTemplate.insert(INSERT_QUERY, params);
        object.setId(id);
        return object;
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public CampaignResource read(@NotNull final Long id) throws EmptyResultDataAccessException {
        return jdbcTemplate.queryForOptional(SELECT_BY_ID, ImmutableMap.of("id", id), (rs, rowNum) -> mapRow(rs))
                .orElseThrow(() -> new EmptyResultDataAccessException("No campaign resource with id " + id, 1));
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Map<Long, CampaignResource> read(@NotNull final Collection<Long> ids) {
        if (ids.isEmpty()) {
            return Collections.emptyMap();
        }
        final HashSet<Long> idsSet = new HashSet<>(ids);
        final Set<CampaignResource> campaignResources = jdbcTemplate.queryForSet(SELECT_BY_IDS,
                ImmutableMap.of("ids", idsSet), (rs, rowNum) -> mapRow(rs));
        final Set<Long> foundIds = campaignResources.stream().map(LongIndexBase::getId).collect(Collectors.toSet());
        final Set<Long> missingIds = Sets.difference(idsSet, foundIds);
        if (!missingIds.isEmpty()) {
            throw new EmptyResultDataAccessException("No campaign resources with ids "
                    + missingIds.stream().map(Objects::toString).collect(Collectors.joining(", ")), idsSet.size());
        }
        return campaignResources.stream().collect(Collectors.toMap(LongIndexBase::getId, Function.identity()));
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public boolean update(@NotNull final CampaignResource object) {
        final Map<String, Object> params = getUpdateParams(object);
        return jdbcTemplate.update(UPDATE_QUERY, params) > 0;
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public boolean updateAll(@NotNull final Collection<CampaignResource> objects) {
        if (objects.isEmpty()) {
            return true;
        }
        final int[] updatedArray = jdbcTemplate.batchUpdate(UPDATE_QUERY,
                objects.stream().map(this::getUpdateParams).collect(Collectors.toList()));
        boolean result = false;
        for (final int updated : updatedArray) {
            result |= updated > 0;
        }
        return result;
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public boolean delete(@NotNull final CampaignResource object) {
        return jdbcTemplate.update(DELETE_BY_ID, ImmutableMap.of("id", object.getId())) > 0;
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public boolean contains(@NotNull final Long id) {
        return jdbcTemplate.queryForObject(EXISTS_BY_ID_QUERY, ImmutableMap.of("id", id), Long.class) > 0;
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public boolean deleteAll(@NotNull final Collection<CampaignResource> objects) {
        if (objects.isEmpty()) {
            return true;
        }
        return jdbcTemplate.update(DELETE_BY_IDS, ImmutableMap.of("ids", objects.stream()
                .map(LongIndexBase::getId).collect(Collectors.toSet()))) > 0;
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public List<CampaignResource> getByCampaignId(final long campaignId) {
        return jdbcTemplate.query(SELECT_BY_CAMPAIGN_ID,
                ImmutableMap.of("campaign_id", campaignId), (rs, rowNum) -> mapRow(rs));
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public List<CampaignResource> getByResourceId(final long resourceId) {
        return jdbcTemplate.query(SELECT_BY_RESOURCE_ID,
                ImmutableMap.of("resource_id", resourceId), (rs, rowNum) -> mapRow(rs));
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Map<Long, Set<Long>> getAvailableCampaignsForServices(@NotNull final Set<Service> services) {
        if (services.isEmpty()) {
            return Collections.emptyMap();
        }
        final Set<Long> serviceIds = services.stream().map(LongIndexBase::getId).collect(Collectors.toSet());
        final Map<Long, Set<Long>> result = new HashMap<>();
        jdbcTemplate.query(SELECT_CAMPAIGNS_BY_SERVICE_IDS, ImmutableMap.of("service_ids", serviceIds), rs -> {
            final long campaignId = rs.getLong("campaign_id");
            final long serviceId = rs.getLong("service_id");
            result.computeIfAbsent(serviceId, k -> new HashSet<>()).add(campaignId);
        });
        services.forEach(s -> result.computeIfAbsent(s.getId(), k -> new HashSet<>()));
        return result;
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Map<Long, Set<Long>> getAvailableCampaignsForResources(@NotNull final Set<Resource> resources) {
        if (resources.isEmpty()) {
            return Collections.emptyMap();
        }
        final Set<Long> resourceIds = resources.stream().map(LongIndexBase::getId).collect(Collectors.toSet());
        final Map<Long, Set<Long>> result = new HashMap<>();
        jdbcTemplate.query(SELECT_CAMPAIGNS_BY_RESOURCE_IDS, ImmutableMap.of("resource_ids", resourceIds), rs -> {
            final long campaignId = rs.getLong("campaign_id");
            final long resourceId = rs.getLong("resource_id");
            result.computeIfAbsent(resourceId, k -> new HashSet<>()).add(campaignId);
        });
        resources.forEach(r -> result.computeIfAbsent(r.getId(), k -> new HashSet<>()));
        return result;
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public Map<Long, Set<Long>> getAvailableCampaignsForSegmentations(@NotNull final Set<Segmentation> segmentations) {
        if (segmentations.isEmpty()) {
            return Collections.emptyMap();
        }
        final Set<Long> segmentationIds = segmentations.stream().map(LongIndexBase::getId).collect(Collectors.toSet());
        final Map<Long, Set<Long>> result = new HashMap<>();
        jdbcTemplate.query(SELECT_CAMPAIGNS_BY_SEGMENTATION_IDS, ImmutableMap.of("segmentation_ids", segmentationIds), rs -> {
            final long campaignId = rs.getLong("campaign_id");
            final long segmentationId = rs.getLong("segmentation_id");
            result.computeIfAbsent(segmentationId, k -> new HashSet<>()).add(campaignId);
        });
        segmentations.forEach(s -> result.computeIfAbsent(s.getId(), k -> new HashSet<>()));
        return result;
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public boolean existsByResource(@NotNull final Resource resource) {
        return jdbcTemplate.queryForObject(EXISTS_BY_RESOURCE_ID_QUERY,
                ImmutableMap.of("resource_id", resource.getId()), Long.class) > 0;
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public boolean existsByService(@NotNull final Service service) {
        return jdbcTemplate.queryForObject(EXISTS_BY_SERVICE_ID_QUERY,
                ImmutableMap.of("service_id", service.getId()), Long.class) > 0;
    }

    @NotNull
    private Map<String, Object> getUpdateParams(@NotNull final CampaignResource object) {
        final Map<String, Object> result = new HashMap<>();
        result.put("campaign_id", object.getCampaignId());
        result.put("resource_id", object.getResourceId());
        result.put("required", object.isRequired());
        result.put("default", object.isDefaultResource());
        result.put("default_amount", object.getDefaultAmount().orElse(null));
        result.put("settings", SqlUtils.toJsonb(object.getSettings()));
        result.put("id", object.getId());
        return result;
    }

    @NotNull
    private CampaignResource mapRow(@NotNull final ResultSet rs) throws SQLException {
        final long id = rs.getLong("id");
        final long campaignId = rs.getLong("campaign_id");
        final long resourceId = rs.getLong("resource_id");
        final boolean required = rs.getBoolean("required");
        final boolean defaultResource = rs.getBoolean("default");
        final Long defaultAmount = getLong(rs, "default_amount");
        final CampaignResource.Settings settings = SqlUtils
                .fromJsonb((PGobject) rs.getObject("settings"), CampaignResource.Settings.class);
        final CampaignResource campaignResource = new CampaignResource(campaignId, resourceId, required,
                defaultResource, defaultAmount, settings);
        campaignResource.setId(id);
        return campaignResource;
    }

}
