package ru.yandex.qe.dispenser.domain.dao.delivery;

import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.dao.DuplicateKeyException;

import ru.yandex.qe.dispenser.domain.resources_model.QuotaRequestDelivery;
import ru.yandex.qe.dispenser.domain.resources_model.QuotaRequestDeliveryResolveStatus;

public class DeliveryDaoImpl implements DeliveryDao {

    @NotNull
    private final Map<Key, QuotaRequestDelivery> deliveryMap =  new ConcurrentHashMap<>();

    @Override
    public List<QuotaRequestDelivery> getUnresolvedByQuotaRequestIdAndProviderId(long quotaRequestId, long providerId) {
        return deliveryMap.values().stream()
                .filter(d -> d.getQuotaRequestId() == quotaRequestId && d.getProviderId() == providerId
                        && !(d.getResolveStatus() == QuotaRequestDeliveryResolveStatus.RESOLVED))
                .collect(Collectors.toList());
    }

    @Override
    public QuotaRequestDelivery readForUpdate(UUID id) {
        final Key key = new Key(id);
        return Optional.ofNullable(deliveryMap.get(key))
                .orElseThrow(() -> new IllegalArgumentException("QuotaRequestDelivery with key " + key + " not found!"));
    }

    @Override
    public List<QuotaRequestDelivery> readAll() {
        return List.copyOf(deliveryMap.values());
    }

    @Override
    public List<QuotaRequestDelivery> getByRequestIds(Set<Long> quotaRequestIds) {
        if (quotaRequestIds.isEmpty()) {
            return List.of();
        }
        return deliveryMap.values().stream().filter(v -> quotaRequestIds.contains(v.getQuotaRequestId()))
                .collect(Collectors.toList());
    }

    @Override
    public List<QuotaRequestDelivery> getUnresolved(long limit, @Nullable UUID fromId) {
        if (limit <= 0) {
            return List.of();
        }
        if (fromId == null) {
            return deliveryMap.values().stream().filter(v -> !(v.getResolveStatus() == QuotaRequestDeliveryResolveStatus.RESOLVED))
                    .sorted(Comparator.comparing(v -> v.getId().toString())).limit(limit).collect(Collectors.toList());
        } else {
            return deliveryMap.values().stream().filter(v -> !(v.getResolveStatus() == QuotaRequestDeliveryResolveStatus.RESOLVED))
                    .sorted(Comparator.comparing(v -> v.getId().toString()))
                    .filter(v -> v.getId().toString().compareTo(fromId.toString()) > 0)
                    .limit(limit).collect(Collectors.toList());
        }
    }

    @Override
    public void create(QuotaRequestDelivery delivery) {
        Key key = Key.from(delivery);
        if (deliveryMap.containsKey(key)) {
            throw new DuplicateKeyException("QuotaRequestDelivery with key " + key + " already exists!");
        }
        deliveryMap.put(key, delivery);
    }

    @Override
    public void update(QuotaRequestDelivery delivery) {
        Key key = Key.from(delivery);
        if (deliveryMap.computeIfPresent(key, (k, d) -> delivery) == null) {
            throw new IllegalArgumentException("QuotaRequestDelivery with key " + key + " not exists to update!");
        }
    }

    @Override
    public void remove(QuotaRequestDelivery delivery) {
        deliveryMap.remove(Key.from(delivery));
    }

    @Override
    public void clear() {
        deliveryMap.clear();
    }

    private static class Key {
        private final UUID id;

        public Key(UUID id) {
            this.id = id;
        }

        public static Key from(QuotaRequestDelivery delivery) {
            return new Key(delivery.getId());
        }

        public UUID getId() {
            return id;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Key key = (Key) o;
            return Objects.equals(id, key.id);
        }

        @Override
        public int hashCode() {
            return Objects.hash(id);
        }

        @Override
        public String toString() {
            return "Key{id=" + id + '}';
        }
    }
}
