package ru.yandex.qe.dispenser.domain.dao.entity;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.base.MoreObjects;
import com.google.common.collect.HashBasedTable;
import com.google.common.collect.Table;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.qe.dispenser.domain.Entity;
import ru.yandex.qe.dispenser.domain.EntitySpec;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.dao.InMemoryKeyDao;
import ru.yandex.qe.dispenser.domain.dao.InMemoryLongKeyDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.quota.QuotaDao;
import ru.yandex.qe.dispenser.domain.support.EntityUsageDiff;
import ru.yandex.qe.dispenser.domain.util.Page;

import static ru.yandex.qe.dispenser.domain.util.ValidationUtils.validateEntityKey;

public class EntityDaoImpl extends InMemoryLongKeyDaoImpl<Entity>
        implements IntegratedEntityDao, InMemoryKeyDao.Normalized<Entity, Entity.Key> {
    @NotNull
    private final Table<Entity, Project, Integer> relations = HashBasedTable.create();

    @Autowired
    private QuotaDao quotaDao;

    @NotNull
    @Override
    public Entity create(final @NotNull Entity newInstance) {
        validateEntityKey(newInstance.getKey().getPublicKey());
        return super.create(addCreationTimeIfAbsent(newInstance));
    }

    @NotNull
    @Override
    public Entity createIfAbsent(final @NotNull Entity obj) {
        validateEntityKey(obj.getKey().getPublicKey());
        return super.createIfAbsent(addCreationTimeIfAbsent(obj));
    }

    @NotNull
    private static Entity addCreationTimeIfAbsent(@NotNull final Entity entity) {
        return entity.hasCreationTime() ? entity :
                Entity.builder(entity)
                        .creationTime(System.currentTimeMillis())
                        .build();
    }

    @Override
    public boolean delete(@NotNull final Entity entity) {
        final boolean deleted = super.delete(entity);
        relations.row(entity).clear();
        return deleted;
    }

    @NotNull
    @Override
    public Set<Entity> filter(@NotNull final Collection<EntitySpec> specs, @NotNull final EntityFilteringParams params) {
        return specs.stream()
                .flatMap(spec -> filter(e -> e.getSpec().equals(spec))
                        .filter(e -> matchesTimeInterval(e, params.getCreatedFrom(), params.getCreatedTo()))
                        .map(e -> addAbsentResources(e, spec))
                        .skip(MoreObjects.firstNonNull(params.getOffset(), 0))
                        .limit(MoreObjects.firstNonNull(params.getLimit(), Integer.MAX_VALUE)))
                .filter(e -> !params.trashOnly() || relations.row(e).isEmpty())
                .collect(Collectors.toSet());
    }

    @NotNull
    @Override
    public Page<Entity> filterPage(@NotNull final EntitySpec spec, @NotNull final EntityFilteringParams params) {

        final Stream<@NotNull Entity> items = filter(e -> e.getSpec().equals(spec))
                .filter(e -> matchesTimeInterval(e, params.getCreatedFrom(), params.getCreatedTo()))
                .filter(e -> !params.trashOnly() || relations.row(e).isEmpty())
                .map(e -> addAbsentResources(e, spec))
                .skip(MoreObjects.firstNonNull(params.getOffset(), 0))
                .limit(MoreObjects.firstNonNull(params.getLimit(), Integer.MAX_VALUE));

        final long totalCount = filter(e -> e.getSpec().equals(spec))
                .filter(e -> matchesTimeInterval(e, params.getCreatedFrom(), params.getCreatedTo()))
                .filter(e -> !params.trashOnly() || relations.row(e).isEmpty())
                .count();

        return Page.of(items, totalCount);
    }

    private static boolean matchesTimeInterval(@NotNull final Entity e, @Nullable final Long start, @Nullable final Long end) {
        final long creationTime = Objects.requireNonNull(e.getCreationTime());
        return (start == null || creationTime >= start) && (end == null || creationTime <= end);
    }

    @NotNull
    private static Entity addAbsentResources(@NotNull final Entity e, @NotNull final EntitySpec realSpec) {
        final Entity.Builder b = Entity.builder(e.getKey()).spec(realSpec);
        Entity.ResourceKey.getKeysForEntitySpec(realSpec)
                .forEach(r -> b.dimension(r, e.getSize(r)));
        return b.build();
    }

    @NotNull
    @Override
    public Table<Entity, Project, Integer> getUsages(@NotNull final Collection<Entity> entities) {
        final Table<Entity, Project, Integer> usages = HashBasedTable.create();
        for (final Entity entity : entities) {
            relations.row(entity).forEach((p, u) -> usages.put(entity, p, u));
        }
        return usages;
    }

    @Override
    public boolean changeUsages(@NotNull final List<EntityUsageDiff> usageDiffs) {
        if (usageDiffs.isEmpty()) {
            return false;
        }
        usageDiffs.forEach(ud -> {
            final Entity e = ud.getEntity();
            final Project p = ud.getProject();
            final int prevUsage = MoreObjects.firstNonNull(relations.get(e, p), 0);
            final int nextUsage = prevUsage + ud.getUsages();
            relations.put(e, p, nextUsage);
        });
        return true;
    }

    @Override
    public boolean cleanIfNeeded(@NotNull final Collection<Entity> entities) {
        for (final Entity entity : entities) {
            final Set<Project> negativeProjects = new HashSet<>();
            relations.row(entity).forEach((p, u) -> {
                if (u <= 0) {
                    negativeProjects.add(p);
                }
            });
            negativeProjects.forEach(project -> relations.remove(entity, project));
        }
        return true;
    }

    @NotNull
    @Override
    public QuotaDao getQuotaDao() {
        return quotaDao;
    }
}
