package ru.yandex.qe.dispenser.domain.dao.entity;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import com.google.common.collect.HashBasedTable;
import com.google.common.collect.Table;
import org.jetbrains.annotations.NotNull;

import ru.yandex.qe.dispenser.api.v1.DiQuotaChangeCause;
import ru.yandex.qe.dispenser.domain.Entity;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.support.EntityOperation;
import ru.yandex.qe.dispenser.domain.support.EntityUsageDiff;
import ru.yandex.qe.dispenser.domain.support.QuotaDiff;
import ru.yandex.qe.dispenser.domain.util.CollectionUtils;
import ru.yandex.qe.dispenser.domain.util.MathUtils;

enum EntityDaoUtils {
    ;

    @NotNull
    static BatchDiff processOperation(@NotNull final EntityOperation op, @NotNull final Table<Entity, Project, Integer> allUsages) {
        final Entity e = op.getEntity();
        final DiQuotaChangeCause cause = op.getCause();

        final Map<Project, Integer> nextUsages = allUsages.row(e);
        //noinspection TooBroadScope
        final Map<Project, Integer> prevUsages = new HashMap<>(nextUsages);

        final Table<Resource, Project, Long> prevQuotas = quotas(e, nextUsages);
        op.processOverUsages(nextUsages);
        final Table<Resource, Project, Long> nextQuotas = quotas(e, nextUsages);

        final BatchDiff result = new BatchDiff();
        MathUtils.decrement(new HashMap<>(nextUsages), prevUsages).forEach((p, u) -> {
            result.addUsageDiff(new EntityUsageDiff(op.getMode(), e, p, op.getMetaValues(), u, cause));
        });
        MathUtils.decrementL(nextQuotas, prevQuotas).rowMap().forEach((r, diffs) -> diffs.forEach((p, diff) -> {
            result.addQuotaDiff(new QuotaDiff(r, p, diff, cause, Collections.emptySet()));
        }));
        return result;
    }

    @NotNull
    private static Table<Resource, Project, Long> quotas(@NotNull final Entity entity, @NotNull final Map<Project, Integer> usages) {
        final Table<Resource, Project, Long> quotas = HashBasedTable.create();
        for (final Resource r : entity.getKey().getSpec().getResources()) {
            MathUtils.incrementL(quotas, r, quotas(entity.getSize(r), usages));
        }
        return quotas;
    }

    @NotNull
    private static Map<Project, Long> quotas(final long entitySize, @NotNull final Map<Project, Integer> usages) {
        final Map<Project, Integer> filteredUsages = CollectionUtils.filter(usages, (p, u) -> u > 0);
        final List<Project> projects = new ArrayList<>(filteredUsages.keySet());
        Collections.sort(projects);
        final int[] projectCounts = projects.stream().mapToInt(filteredUsages::get).toArray();

        final long[] projectQuotas = QuotaSharer.INSTANCE.share(entitySize, projectCounts);

        return IntStream.range(0, projectQuotas.length).boxed().collect(Collectors.toMap(projects::get, i -> projectQuotas[i]));
    }
}
