package ru.yandex.qe.dispenser.domain.dao.entity;

import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.collect.HashBasedTable;
import com.google.common.collect.Table;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.dao.IncorrectResultSizeDataAccessException;

import ru.yandex.qe.dispenser.domain.Entity;
import ru.yandex.qe.dispenser.domain.EntitySpec;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.dao.InMemoryKeyDao;
import ru.yandex.qe.dispenser.domain.dao.InMemoryLongKeyDaoImpl;
import ru.yandex.qe.dispenser.domain.dao.quota.QuotaDao;
import ru.yandex.qe.dispenser.domain.distributed.Identifier;
import ru.yandex.qe.dispenser.domain.support.EntityOwnership;
import ru.yandex.qe.dispenser.domain.support.EntityUsageDiff;
import ru.yandex.qe.dispenser.domain.util.Page;

public class InMemoryOnlyEntityDao extends InMemoryLongKeyDaoImpl<Entity>
        implements IntegratedEntityDao, InMemoryKeyDao.Normalized<Entity, Entity.Key> {
    private final Map<Identifier, TimeCountPair> lastUpdates = new ConcurrentHashMap<>();

    @Autowired
    private QuotaDao quotaDao;

    private final ConcurrentMap<Entity, Project> relations = new ConcurrentHashMap<>();

    private final ConcurrentSkipListSet<Entity> entityOrder = new ConcurrentSkipListSet<>(new Comparator<Entity>() {
        @Override
        public int compare(final Entity o1, final Entity o2) {
            return 10000 * Long.compare(o1.getCreationTime(), o2.getCreationTime()) + o1.getKey().getPublicKey().compareTo(o2.getKey().getPublicKey());
        }
    });


    @NotNull
    @Override
    public Map<Entity.Key, Entity> readAll(@NotNull final Collection<Entity.Key> keys) {
        final Map<Entity.Key, Entity> result = new HashMap<>(keys.size());
        for (final Entity.Key key : keys) {
            final Entity next = id2obj.get(calcId(key));
            if (next == null) {
                throw new IncorrectResultSizeDataAccessException("No object in " + getClass().getSimpleName() + " with key '" + key + "'!", keys.size());
            }
            result.put(key, next);
        }
        return result;
    }

    @NotNull
    @Override
    public Stream<Entity> filter(@NotNull final Predicate<Entity> predicate) {
        return id2obj.values().stream().filter(predicate);
    }

    @Override
    @NotNull
    public Entity create(@NotNull final Entity entity) {
        return abstractCreate(entity, super::create);
    }

    @Override
    @NotNull
    public Entity createIfAbsent(@NotNull final Entity entity) {
        return abstractCreate(entity, super::createIfAbsent);
    }

    public boolean hasEntity(@NotNull final Entity entity) {
        return id2obj.containsKey(calcId(entity));
    }


    public Entity abstractCreate(@NotNull final Entity entity, @NotNull final Function<Entity, Entity> func) {
        Entity fixed = updateFor(entity);
        Objects.requireNonNull(fixed.getIdentifier());
        fixed = func.apply(fixed);
        entityOrder.add(fixed);
        return fixed;
    }

    @Override
    public boolean delete(final @NotNull Entity entity) {
        Objects.requireNonNull(entity);
        lastUpdates.computeIfPresent(entity.getIdentifier(), new BiFunction<Identifier, TimeCountPair, TimeCountPair>() {
            @Override
            public TimeCountPair apply(final Identifier identifier, final TimeCountPair timeCountPair) {
                return new TimeCountPair(timeCountPair.ts, timeCountPair.count - 1);
            }
        });
        assert entityOrder.contains(entity);
        entityOrder.remove(entity);

        final boolean deleted = super.delete(entity);
        relations.remove(entity);
        return deleted;
    }

    public Map<Identifier, Long> getUpdateTimes() {
        return lastUpdates.entrySet().stream().filter(p -> p.getValue().count > 0).collect(Collectors.toMap(Map.Entry::getKey, p -> p.getValue().ts));
    }

    private Entity updateFor(@NotNull final Entity entity) {
        if (entity.getId() < 0) {
            entity.setId((calcId(entity)));
        }

        final Entity fixedEntity = addCreationTimeIfAbsent(entity);
        assert entity.getId() == fixedEntity.getId();
        Objects.requireNonNull(fixedEntity.getIdentifier());
        lastUpdates.compute(fixedEntity.getIdentifier(), new BiFunction<Identifier, TimeCountPair, TimeCountPair>() {
            @Override
            public TimeCountPair apply(final Identifier identifier, final TimeCountPair TimeCountPair) {
                Objects.requireNonNull(fixedEntity.getCreationTime());
                if (TimeCountPair == null) {
                    return new TimeCountPair(fixedEntity.getCreationTime(), 1);
                }
                return new TimeCountPair(Math.max(fixedEntity.getCreationTime(), TimeCountPair.ts), TimeCountPair.count + 1);
            }
        });
        return fixedEntity;
    }

    private Long calcId(@NotNull final Entity.Key key) {
        return Integer.MAX_VALUE + 2L + (long) key.hashCode();
    }

    private Long calcId(@NotNull final Entity entity) {
        return calcId(entity.getKey());
    }

    public Stream<EntityOwnership> getAllUsagesFrom(@NotNull final Map<Identifier, Long> timestamps) {
        return entityOrder.stream().filter(entity -> {
            final Identifier identifier = entity.getIdentifier();
            Objects.requireNonNull(entity.getCreationTime(), "creation time should present");
            return !timestamps.containsKey(identifier) || entity.getCreationTime() > timestamps.get(identifier);
        }).map(e -> {
            final Project p = relations.get(e);
            final int usages;
            if (p != null) {
                assert p.isPersonal();
                return Optional.of(new EntityOwnership(e, p, 1));
            }
            return Optional.<EntityOwnership>empty();

        }).filter(Optional::isPresent).map(Optional::get);
    }

    @NotNull
    @Override
    public Set<Entity> filter(@NotNull final Collection<EntitySpec> specs, @NotNull final EntityFilteringParams params) {
        return filterStream(specs, params).collect(Collectors.toSet());
    }

    @NotNull
    @Override
    public Stream<Entity> filterStream(@NotNull final Collection<EntitySpec> specs, @NotNull final EntityFilteringParams params) {
        return specs.stream()
                .flatMap(spec -> filter(e -> e.getSpec().equals(spec))
                        .filter(e -> matchesTimeInterval(e, params.getCreatedFrom(), params.getCreatedTo()))
                        .map(e -> addAbsentResources(e, spec))
                        .skip(Optional.ofNullable(params.getOffset()).orElse(0))
                        .limit(Optional.ofNullable(params.getLimit()).orElse(Integer.MAX_VALUE))) // TODO limit: is it correct?
                .filter(e -> !params.trashOnly() || relations.containsKey(e));
    }

    @NotNull
    @Override
    public Page<Entity> filterPage(@NotNull final EntitySpec spec, @NotNull final EntityFilteringParams params) {
        final Stream<@NotNull Entity> items = filter(e -> e.getSpec().equals(spec))
                .filter(e -> matchesTimeInterval(e, params.getCreatedFrom(), params.getCreatedTo()))
                .filter(e -> !params.trashOnly() || relations.containsKey(e))
                .map(e -> addAbsentResources(e, spec))
                .skip(Optional.ofNullable(params.getOffset()).orElse(0))
                .limit(Optional.ofNullable(params.getLimit()).orElse(Integer.MAX_VALUE));// TODO limit: is it correct?

        final long totalCount = filter(e -> e.getSpec().equals(spec))
                .filter(e -> matchesTimeInterval(e, params.getCreatedFrom(), params.getCreatedTo()))
                .filter(e -> !params.trashOnly() || relations.containsKey(e))
                .count();

        return Page.of(items, totalCount);
    }

    private static boolean matchesTimeInterval(@NotNull final Entity e, @Nullable final Long start, @Nullable final Long end) {
        final long creationTime = Objects.requireNonNull(e.getCreationTime());
        return (start == null || creationTime >= start) && (end == null || creationTime <= end);
    }

    @NotNull
    private static Entity addAbsentResources(@NotNull final Entity e, @NotNull final EntitySpec realSpec) {
        final Entity.Builder b = Entity.builder(e.getKey())
                .id(e.getId())
                .identifier(e.getIdentifier())
                .creationTime(e.getCreationTime())
                .spec(realSpec);
        for (final Resource r : realSpec.getResources()) {
            b.dimension(r, e.getSize(r));
        }
        return b.build();
    }

    @NotNull
    @Override
    public Table<Entity, Project, Integer> getUsages(@NotNull final Collection<Entity> entities) {
        final Table<Entity, Project, Integer> usages = HashBasedTable.create();
        entities.forEach(e -> {
            final Project p = relations.get(e);
            if (p != null) {
                usages.put(e, relations.get(e), 1);
            }
        });
        return usages;
    }

    @Override
    public boolean changeUsages(@NotNull final List<EntityUsageDiff> usageDiffs) {
        if (usageDiffs.isEmpty()) {
            return false;
        }
        usageDiffs.forEach(ud -> {
            final Entity e = ud.getEntity();
            final Project p = ud.getProject();

            final int cu = ud.getUsages();

            switch (cu) {
                case -1:
                    if (!relations.containsKey(e) || !p.equals(relations.get(e))) {
                        throw new RuntimeException("no usages found for reducing");
                    }
                    final boolean result = relations.remove(e, p);
                    assert result;
                    break;
                case 1:
                    relations.compute(e, new BiFunction<Entity, Project, Project>() {
                        @Override
                        public Project apply(final Entity entity, final Project project) {
                            if (project != null) {
                                throw new RuntimeException("can't add usages,sharing is not supported");
                            }
                            return p;
                        }
                    });
                    break;
                case 0:
                    break;
                default:
                    throw new RuntimeException("can't change usages on " + cu + " sharing is not supported");
            }
        });
        return true;
    }

    @Override
    public boolean cleanIfNeeded(@NotNull final Collection<Entity> entities) {
        entities.forEach(e -> {
            if (relations.containsKey(e)) {
                throw new RuntimeException("removed entity presents");
            }
        });
        return true;
    }

    @NotNull
    @Override
    public QuotaDao getQuotaDao() {
        return quotaDao;
    }

    @NotNull
    private static Entity addCreationTimeIfAbsent(@NotNull final Entity entity) {
        return entity.hasCreationTime() ? entity :
                Entity.builder(entity)
                        .id(entity.getId())
                        .identifier(entity.getIdentifier())
                        .creationTime(System.currentTimeMillis())
                        .build();
    }


    private static class TimeCountPair {
        final long ts;
        final long count;

        private TimeCountPair(final long ts, final long count) {
            this.ts = ts;
            this.count = count;
        }
    }
}
