package ru.yandex.qe.dispenser.domain.dao.entity.spec;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.SqlParameterSource;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.domain.EntitySpec;
import ru.yandex.qe.dispenser.domain.Resource;
import ru.yandex.qe.dispenser.domain.dao.SqlDaoBase;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.index.LongIndexBase;

import static ru.yandex.qe.dispenser.domain.dao.entity.SqlEntityDaoUtils.entityCreationTimeIndexName;
import static ru.yandex.qe.dispenser.domain.dao.entity.SqlEntityDaoUtils.entityExpirationTimeIndexName;
import static ru.yandex.qe.dispenser.domain.dao.entity.SqlEntityDaoUtils.formatEntityTable;
import static ru.yandex.qe.dispenser.domain.dao.entity.SqlEntityDaoUtils.formatEntityUsageTable;
import static ru.yandex.qe.dispenser.domain.dao.entity.SqlEntityDaoUtils.toEntityTableName;
import static ru.yandex.qe.dispenser.domain.dao.entity.SqlEntityDaoUtils.toEntityUsageTableName;

public class SqlEntitySpecDao extends SqlDaoBase implements EntitySpecDao {
    private static final String GET_QUERY = "SELECT entity_spec.*, array_agg(entity_spec_resource.resource_id) FROM entity_spec, entity_spec_resource WHERE entity_spec.id = entity_spec_resource.entity_spec_id";
    private static final String GROUP_BY_SUFFIX = "GROUP BY entity_spec.id";

    private static final String CREATE_QUERY = "INSERT INTO entity_spec (key, service_id, tag, description, expirable) VALUES (:key, :serviceId, :tag, :description, :expirable)";
    private static final String CREATE_RELATIONS_QUERY = "INSERT INTO entity_spec_resource (entity_spec_id, resource_id) VALUES (:entitySpecId, :resourceId)";
    private static final String CREATE_ENTITY_TABLE_QUERY = "CREATE TABLE %s ("
            + "  id            bigserial PRIMARY KEY,"
            + "  key           short_text NOT NULL,"
            + "  dimensions    jsonb NOT NULL,"
            + "  creation_time timestamp NOT NULL DEFAULT now(),"
            + "  ${expiration_time}"
            + "  UNIQUE (key)"
            + ")";
    private static final String CREATE_ENTITY_USAGE_TABLE_QUERY = "CREATE TABLE %s ("
            + "  entity_id     bigint REFERENCES %s(id) ON UPDATE CASCADE ON DELETE CASCADE,"
            + "  project_id    bigint REFERENCES project(id) ON UPDATE CASCADE ON DELETE CASCADE,"
            + "  usages        int NOT NULL,"
            + "  PRIMARY KEY (entity_id, project_id)"
            + ")";
    private static final String CREATE_ENTITY_CREATION_TIME_INDEX = "CREATE INDEX %s ON %s (creation_time);";
    private static final String CREATE_ENTITY_EXPIRATION_TIME_INDEX = "CREATE INDEX %s ON %s (expiration_time);";

    private static final String READ_BY_ID_QUERY = GET_QUERY + " AND entity_spec.id = :id " + GROUP_BY_SUFFIX;
    private static final String READ_BY_KEY_QUERY = GET_QUERY + " AND entity_spec.key = :key AND entity_spec.service_id = :serviceId " + GROUP_BY_SUFFIX;
    private static final String UPDATE_QUERY = "UPDATE entity_spec SET key = :key, service_id = :serviceId, description = :description, tag = :tag WHERE id = :id";
    private static final String UPDATE_RESOURCE_RELATIONS_QUERY = "UPDATE entity_spec_resource SET resource_id = :resourceId WHERE entity_spec_id = :id";

    private static final String DELETE_QUERY = "DELETE FROM entity_spec WHERE id = (:id)";
    private static final String DROP_TABLE_QUERY = "DROP TABLE %s CASCADE";
    private static final String CLEAR_QUERY = "TRUNCATE entity_spec CASCADE";

    @NotNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED) // for postgres test
    public Set<EntitySpec> getAll() {
        return jdbcTemplate.queryForSet(GET_QUERY + " " + GROUP_BY_SUFFIX, this::toEntitySpec);
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public EntitySpec create(@NotNull final EntitySpec entitySpec) {
        entitySpec.setId(jdbcTemplate.insert(CREATE_QUERY, toParams(entitySpec)));

        final String createTableTemplate = CREATE_ENTITY_TABLE_QUERY.replace("${expiration_time}", entitySpec.isExpirable() ? "expiration_time timestamp," : "");
        jdbcTemplate.update(formatEntityTable(createTableTemplate, entitySpec));
        jdbcTemplate.update(String.format(CREATE_ENTITY_CREATION_TIME_INDEX, entityCreationTimeIndexName(entitySpec), toEntityTableName(entitySpec)));
        if (entitySpec.isExpirable()) {
            jdbcTemplate.update(String.format(CREATE_ENTITY_EXPIRATION_TIME_INDEX, entityExpirationTimeIndexName(entitySpec), toEntityTableName(entitySpec)));
        }

        jdbcTemplate.update(String.format(CREATE_ENTITY_USAGE_TABLE_QUERY, toEntityUsageTableName(entitySpec), toEntityTableName(entitySpec)));

        addRelations(entitySpec, entitySpec.getResources());

        return entitySpec;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public boolean addRelations(@NotNull final EntitySpec spec, @NotNull final Collection<Resource> resources) {
        final SqlParameterSource[] resourceRelations = resources.stream()
                .map(LongIndexBase::getId)
                .map(resourceId -> new MapSqlParameterSource("entitySpecId", spec.getId()).addValue("resourceId", resourceId))
                .toArray(SqlParameterSource[]::new);
        return jdbcTemplate.batchUpdate(CREATE_RELATIONS_QUERY, resourceRelations).length > 0;
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED)  // only for tests
    public EntitySpec read(@NotNull final Long id) throws EmptyResultDataAccessException {
        return jdbcTemplate.queryForOptional(READ_BY_ID_QUERY, Collections.singletonMap("id", id), this::toEntitySpec)
                .orElseThrow(() -> new EmptyResultDataAccessException("No entity spec with id " + id, 1));
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public EntitySpec read(@NotNull final EntitySpec.Key key) throws EmptyResultDataAccessException {
        final Map<String, ?> params = ImmutableMap.of("key", key.getPublicKey(), "serviceId", key.getService().getId());
        return jdbcTemplate.queryForOptional(READ_BY_KEY_QUERY, params, this::toEntitySpec)
                .orElseThrow(() -> new EmptyResultDataAccessException("No entity spec with key " + key, 1));
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public boolean update(@NotNull final EntitySpec entitySpec) {
        boolean updated = jdbcTemplate.update(UPDATE_QUERY, toParams(entitySpec)) > 0;
        final SqlParameterSource[] insertions = entitySpec.getResources().stream()
                .map(LongIndexBase::getId)
                .map(id -> Collections.singletonMap("resourceId", id))
                .toArray(SqlParameterSource[]::new);
        updated |= jdbcTemplate.batchUpdate(UPDATE_RESOURCE_RELATIONS_QUERY, insertions).length > 0;  // TODO
        return updated;
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public boolean delete(@NotNull final EntitySpec entitySpec) {
        boolean deleted = jdbcTemplate.update(formatEntityUsageTable(DROP_TABLE_QUERY, entitySpec)) > 0;
        deleted |= jdbcTemplate.update(formatEntityTable(DROP_TABLE_QUERY, entitySpec)) > 0;
        deleted |= jdbcTemplate.update(DELETE_QUERY, "id", entitySpec.getId()) > 0;
        return deleted;
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public boolean clear() {
        final EntitySpec[] allSpecs = getAll().stream().toArray(EntitySpec[]::new);
        boolean cleared = false;
        if (allSpecs.length > 0) {
            cleared = jdbcTemplate.update(formatEntityUsageTable(DROP_TABLE_QUERY, allSpecs)) > 0;
            cleared |= jdbcTemplate.update(formatEntityTable(DROP_TABLE_QUERY, allSpecs)) > 0;
        }
        cleared |= jdbcTemplate.update(CLEAR_QUERY) > 0;
        return cleared;
    }

    @NotNull
    private EntitySpec toEntitySpec(@NotNull final ResultSet rs, final int i) throws SQLException {
        final Set<Resource> resources = Arrays.stream((Long[]) (rs.getArray("array_agg")).getArray())
                .map(resourceId -> Hierarchy.get().getResourceReader().read(resourceId))
                .collect(Collectors.toSet());
        final long id = rs.getLong("id");
        final EntitySpec entitySpec = EntitySpec.builder()
                .withKey(rs.getString("key"))
                .withDescription(rs.getString("description"))
                .withCustomTag(rs.getString("tag"))
                .overResources(resources)
                .expirable(rs.getBoolean("expirable"))
                .build();
        entitySpec.setId(id);
        return entitySpec;
    }

    @NotNull
    private Map<String, ?> toParams(@NotNull final EntitySpec spec) {
        return ImmutableMap.<String, Object>builder()
                .put("key", spec.getKey().getPublicKey())
                .put("serviceId", spec.getService().getId())
                .put("description", spec.getDescription())
                .put("expirable", spec.isExpirable())
                .put("tag", spec.getTag())
                .build();
    }
}
