package ru.yandex.qe.dispenser.domain.dao.group;

import java.util.Collection;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.Sets;
import org.jetbrains.annotations.NotNull;
import org.springframework.dao.EmptyResultDataAccessException;

import ru.yandex.qe.dispenser.domain.YaGroup;
import ru.yandex.qe.dispenser.domain.dao.InMemoryKeyDao;

public interface GroupDao extends InMemoryKeyDao.Normalized<YaGroup, String> {

    void createIfAbsent(@NotNull final Collection<YaGroup> groups);

    @NotNull
    Optional<YaGroup> tryReadYaGroupByStaffId(final long staffId);

    @NotNull
    Set<YaGroup> tryReadYaGroupsByStaffIds(@NotNull final Collection<Long> staffIds);

    @NotNull
    Optional<YaGroup> tryReadYaGroupByUrl(@NotNull final String url);

    @NotNull
    Set<YaGroup> tryReadYaGroupsByUrls(@NotNull final Collection<String> urls);

    @NotNull
    default YaGroup readYaGroupByUrl(@NotNull final String url) {
        return tryReadYaGroupByUrl(url).orElseThrow(() -> new EmptyResultDataAccessException("No group with url " + url, 1));
    }

    @NotNull
    default Set<YaGroup> readYaGroupsByUrls(@NotNull final Collection<String> urls) {
        final Set<YaGroup> groups = tryReadYaGroupsByUrls(urls);
        final Set<String> foundUrls = groups.stream().map(YaGroup::getUrl).collect(Collectors.toSet());
        final Set<String> missingUrls = Sets.difference(new HashSet<>(urls), foundUrls);
        if (!missingUrls.isEmpty()) {
            throw new EmptyResultDataAccessException("No groups with urls " + missingUrls, urls.size());
        }
        return groups;
    }

    @NotNull
    default YaGroup readYaGroupByStaffId(final long staffId) {
        return tryReadYaGroupByStaffId(staffId).orElseThrow(() -> new EmptyResultDataAccessException("No group with staff id " + staffId, 1));
    }

    @NotNull
    default Set<YaGroup> readYaGroupsByStaffIds(@NotNull final Collection<Long> staffIds) {
        final Set<YaGroup> groups = tryReadYaGroupsByStaffIds(staffIds);
        final Set<Long> foundStaffIds = groups.stream().map(YaGroup::getStaffId).collect(Collectors.toSet());
        final Set<Long> missingStaffIds = Sets.difference(new HashSet<>(staffIds), foundStaffIds);
        if (!missingStaffIds.isEmpty()) {
            throw new EmptyResultDataAccessException("No groups with staff ids " + missingStaffIds, staffIds.size());
        }
        return groups;
    }

}
