package ru.yandex.qe.dispenser.domain.dao.group;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import org.jetbrains.annotations.NotNull;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.qe.dispenser.domain.YaGroup;
import ru.yandex.qe.dispenser.domain.dao.SqlDaoBase;

public class SqlGroupDao extends SqlDaoBase implements GroupDao {
    private static final String GET_ALL_QUERY = "SELECT * FROM yandex_group";

    private static final String CREATE_QUERY = "INSERT INTO yandex_group (url_acceptable_key, group_type, staff_id, deleted) VALUES (:url, CAST(:groupType as yandex_group_type), :staff_id, :deleted)";
    private static final String CREATE_IF_ABSENT_QUERY = "INSERT INTO yandex_group (url_acceptable_key, group_type, staff_id, deleted) VALUES (:url, CAST(:groupType as yandex_group_type), :staff_id, :deleted) ON CONFLICT DO NOTHING";
    private static final String READ_BY_ID_QUERY = "SELECT * FROM yandex_group WHERE id = :id";
    private static final String READ_BY_STAFF_ID_QUERY = "SELECT * FROM yandex_group WHERE staff_id = :staff_id";
    private static final String READ_BY_STAFF_IDS_QUERY = "SELECT * FROM yandex_group WHERE staff_id IN (:staff_ids)";
    private static final String READ_BY_URL_QUERY = "SELECT * FROM yandex_group WHERE url_acceptable_key = :url";
    private static final String READ_BY_URLS_QUERY = "SELECT * FROM yandex_group WHERE url_acceptable_key IN (:urls)";
    private static final String UPDATE_QUERY = "UPDATE yandex_group SET url_acceptable_key = :url, group_type = CAST(:groupType as yandex_group_type), staff_id = :staff_id, deleted = :deleted WHERE id = :groupId";
    private static final String DELETE_QUERY = "DELETE FROM yandex_group WHERE id = :id";
    private static final String CLEAR_QUERY = "TRUNCATE yandex_group CASCADE";

    @NotNull
    @Override
    public Set<YaGroup> getAll() {
        return jdbcTemplate.queryForSet(GET_ALL_QUERY, this::toGroup);
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public YaGroup create(@NotNull final YaGroup group) {
        jdbcTemplate.update(CREATE_QUERY, toCreateParams(group));
        return read(group.getUrl());
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public void createIfAbsent(@NotNull final Collection<YaGroup> groups) {
        if (groups.isEmpty()) {
            return;
        }
        jdbcTemplate.batchUpdate(CREATE_IF_ABSENT_QUERY, groups.stream().map(this::toCreateParams).collect(Collectors.toList()));
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public YaGroup read(@NotNull final Long id) throws EmptyResultDataAccessException {
        return jdbcTemplate.queryForOptional(READ_BY_ID_QUERY, ImmutableMap.of("id", id), this::toGroup)
                .orElseThrow(() -> new EmptyResultDataAccessException("No group with id " + id, 1));
    }

    @NotNull
    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public YaGroup read(@NotNull final String url) throws EmptyResultDataAccessException {
        return readYaGroupByUrl(url);
    }

    @Override
    @NotNull
    @Transactional(propagation = Propagation.REQUIRED)
    public Optional<YaGroup> tryReadYaGroupByStaffId(final long staffId) {
        return jdbcTemplate.queryForOptional(READ_BY_STAFF_ID_QUERY, ImmutableMap.of("staff_id", staffId), this::toGroup);
    }

    @Override
    @NotNull
    @Transactional(propagation = Propagation.REQUIRED)
    public Set<YaGroup> tryReadYaGroupsByStaffIds(@NotNull final Collection<Long> staffIds) {
        if (staffIds.isEmpty()) {
            return Collections.emptySet();
        }
        return jdbcTemplate.queryForSet(READ_BY_STAFF_IDS_QUERY, Collections.singletonMap("staff_ids", staffIds), this::toGroup);
    }

    @Override
    @NotNull
    @Transactional(propagation = Propagation.REQUIRED)
    public Optional<YaGroup> tryReadYaGroupByUrl(@NotNull final String url) {
        return jdbcTemplate.queryForOptional(READ_BY_URL_QUERY, ImmutableMap.of("url", url), this::toGroup);
    }

    @Override
    @NotNull
    @Transactional(propagation = Propagation.REQUIRED)
    public Set<YaGroup> tryReadYaGroupsByUrls(@NotNull final Collection<String> urls) {
        if (urls.isEmpty()) {
            return Collections.emptySet();
        }
        return jdbcTemplate.queryForSet(READ_BY_URLS_QUERY, Collections.singletonMap("urls", urls), this::toGroup);
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public boolean update(@NotNull final YaGroup group) {
        return jdbcTemplate.update(UPDATE_QUERY, toUpdateParams(group)) > 0;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public boolean updateAll(final Collection<YaGroup> groups) {
        if (groups.isEmpty()) {
            return false;
        }
        final int[] updatedArray = jdbcTemplate.batchUpdate(UPDATE_QUERY,
                groups.stream().map(this::toUpdateParams).collect(Collectors.toList()));
        boolean result = false;
        for (final int updated : updatedArray) {
            result |= updated > 0;
        }
        return result;
    }

    @Override
    @Transactional(propagation = Propagation.REQUIRED)
    public boolean delete(@NotNull final YaGroup group) {
        return jdbcTemplate.update(DELETE_QUERY, ImmutableMap.of("id", group.getId())) > 0;
    }

    @Override
    @Transactional(propagation = Propagation.MANDATORY)
    public boolean clear() {
        return jdbcTemplate.update(CLEAR_QUERY) > 0;
    }

    @NotNull
    private Map<String, ?> toCreateParams(@NotNull final YaGroup group) {
        final Map<String, Object> params = new HashMap<>();
        params.put("url", group.getUrl());
        params.put("groupType", group.getType().name());
        params.put("staff_id", group.getStaffId());
        params.put("deleted", group.isDeleted());
        return params;
    }

    @NotNull
    private Map<String, ?> toUpdateParams(@NotNull final YaGroup group) {
        final Map<String, Object> params = new HashMap<>();
        params.put("url", group.getUrl());
        params.put("groupType", group.getType().name());
        params.put("staff_id", group.getStaffId());
        params.put("deleted", group.isDeleted());
        params.put("groupId", group.getId());
        return params;
    }

}
