package ru.yandex.qe.dispenser.domain.dao.history.quota;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.time.Instant;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import com.healthmarketscience.sqlbuilder.SelectQuery;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;

import ru.yandex.qe.dispenser.api.v1.DiOrder;
import ru.yandex.qe.dispenser.domain.Project;
import ru.yandex.qe.dispenser.domain.Quota;
import ru.yandex.qe.dispenser.domain.QuotaHistoryEvent;
import ru.yandex.qe.dispenser.domain.QuotaSpec;
import ru.yandex.qe.dispenser.domain.Segment;
import ru.yandex.qe.dispenser.domain.dao.SqlDaoBase;
import ru.yandex.qe.dispenser.domain.dao.history.HistoryFilter;
import ru.yandex.qe.dispenser.domain.dao.quota.QuotaKeyFilter;
import ru.yandex.qe.dispenser.domain.dao.quota.QuotaReader;
import ru.yandex.qe.dispenser.domain.dao.quota.SqlQuotaDao;
import ru.yandex.qe.dispenser.domain.dao.quota.segment.SqlQuotaSegmentDao;
import ru.yandex.qe.dispenser.domain.hierarchy.Hierarchy;
import ru.yandex.qe.dispenser.domain.util.HistoryDaoUtils;
import ru.yandex.qe.dispenser.domain.util.RelativePage;
import ru.yandex.qe.dispenser.domain.util.RelativePageInfo;

import static ru.yandex.qe.dispenser.domain.util.HistoryDaoUtils.operatorByOrder;
import static ru.yandex.qe.dispenser.domain.util.HistoryDaoUtils.toCondition;

public class SqlQuotaHistoryDao extends SqlDaoBase implements QuotaHistoryDao {
    private static final String LIMIT = " LIMIT :pageSize";
    private static final String ORDER = " ORDER BY id %s";

    private static final String CREATE_QUERY = "INSERT INTO quota_history(quota_id, person_id, tvm_id, comment, updated, old_max, new_max, old_own_max, new_own_max, issue_key) " +
            "VALUES(:quotaId, :personId, :tvmId, :comment, :updated, :oldMax, :newMax, :oldOwnMax, :newOwnMax, :issueKey)";
    private static final String GET_IDS_BY_QUOTA_IDS_QUERY_TEMPLATE = "SELECT id FROM quota_history WHERE quota_id IN (%s) AND id %s :offset";
    private static final String GET_WITH_QUOTA_KEY_QUERY_TEMPLATE = "SELECT qh.*, q.project_id, q.quota_spec_id, qs.segment_id, qs.segmentation_id FROM quota_history qh " +
            "JOIN quota q ON qh.quota_id = q.id LEFT JOIN quota_segment qs ON qh.quota_id = qs.quota_id WHERE qh.id IN (%s)";
    private static final String CLEAR_QUERY = "TRUNCATE quota_history CASCADE";
    private static final String CLEAR_BEFORE_QUERY = "DELETE FROM quota_history WHERE updated < :moment";

    @Override
    public QuotaHistoryEvent create(final QuotaHistoryEvent event) {
        final long id = jdbcTemplate.insert(CREATE_QUERY, toParams(event));
        event.setId(id);
        return event;
    }

    @Override
    public void create(final Collection<QuotaHistoryEvent> events) {
        if (events.isEmpty()) {
            return;
        }
        jdbcTemplate.batchUpdate(CREATE_QUERY, events.stream().map(SqlQuotaHistoryDao::toParams).collect(Collectors.toList()));
    }

    @Override
    public QuotaHistoryEvent read(final long id) {
        return jdbcTemplate.query(String.format(GET_WITH_QUOTA_KEY_QUERY_TEMPLATE, id), new MapSqlParameterSource(ImmutableMap.of("id", id)), SqlQuotaHistoryDao::processRows).stream()
                .findFirst()
                .orElseThrow(() -> new EmptyResultDataAccessException("No quota history event with id [" + id + "]", 1));
    }

    @Override
    public RelativePage<QuotaHistoryEvent> read(final Quota.Key quotaKey, final HistoryFilter filter, final RelativePageInfo pageInfo, final DiOrder order) {
        final SelectQuery quotaIdQuery = SqlQuotaDao.createIdQuery(new QuotaKeyFilter(quotaKey));
        final String subquery = quotaIdQuery.toString();
        final Map<String, Object> params = new HashMap<>(HistoryDaoUtils.toParams(filter));
        final String eventIdsQuery = String.format(GET_IDS_BY_QUOTA_IDS_QUERY_TEMPLATE, subquery, operatorByOrder(order)) + toCondition(filter, false)
                + orderStatement(order) + LIMIT;
        return getRelativePage(String.format(GET_WITH_QUOTA_KEY_QUERY_TEMPLATE, eventIdsQuery)
                + orderStatement(order), params, SqlQuotaHistoryDao::processRows, pageInfo);
    }

    @Override
    public RelativePage<QuotaHistoryEvent> read(final QuotaReader.QuotaFilterParams filter,
                                                final HistoryFilter historyFilter,
                                                final RelativePageInfo pageInfo,
                                                final DiOrder order) {
        final SelectQuery quotaIdQuery = SqlQuotaDao.createIdQuery(filter);
        final String quotaSubquery = quotaIdQuery.toString();
        final Map<String, Object> params = new HashMap<>(HistoryDaoUtils.toParams(historyFilter));
        final String eventIdsQuery = String.format(GET_IDS_BY_QUOTA_IDS_QUERY_TEMPLATE + toCondition(historyFilter, false) +
                orderStatement(order) + LIMIT, quotaSubquery, operatorByOrder(order));
        return getRelativePage(String.format(GET_WITH_QUOTA_KEY_QUERY_TEMPLATE, eventIdsQuery) +
                orderStatement(order), params, SqlQuotaHistoryDao::processRows, pageInfo);
    }

    @Override
    public boolean clear() {
        return jdbcTemplate.update(CLEAR_QUERY) > 0;
    }

    @Override
    public boolean clearBefore(final Instant moment) {
        return jdbcTemplate.update(CLEAR_BEFORE_QUERY, ImmutableMap.of("moment", Timestamp.from(moment))) > 0;
    }

    private static String orderStatement(final DiOrder order) {
        return String.format(ORDER, order.toString());
    }

    private static Map<String, ?> toParams(final QuotaHistoryEvent event) {
        final Map<String, Object> params = new HashMap<>();
        params.put("quotaId", event.getQuotaId());
        params.put("personId", event.getPersonId());
        params.put("tvmId", event.getTvmId());
        params.put("updated", Timestamp.from(event.getUpdated()));
        params.put("comment", event.getComment());
        params.put("oldMax", event.getOldMax());
        params.put("newMax", event.getNewMax());
        params.put("oldOwnMax", event.getOldOwnMax());
        params.put("newOwnMax", event.getNewOwnMax());
        params.put("issueKey", event.getIssueKey());
        return params;
    }

    private static List<QuotaHistoryEvent> processRows(final ResultSet rs) throws SQLException {
        final Map<Long, QuotaHistoryEvent.Builder> eventBuilderById = new LinkedHashMap<>();
        final Map<Long, Quota.Key> quotaKeyByQuotaId = new HashMap<>();
        while (rs.next()) {
            final QuotaSpec quotaSpec = Hierarchy.get().getQuotaSpecReader().read(rs.getLong("quota_spec_id"));
            final Project project = Hierarchy.get().getProjectReader().read(rs.getLong("project_id"));
            final long quotaId = rs.getLong("quota_id");
            final Quota.Key key = quotaKeyByQuotaId.computeIfAbsent(quotaId, id -> new Quota.Key(quotaSpec, project, new HashSet<>()));
            rs.getLong("segmentation_id");
            if (!rs.wasNull()) {
                final Segment segment = SqlQuotaSegmentDao.toSegment(rs, 0);
                key.getSegments().add(segment);
            }
            final long id = rs.getLong("id");
            if (!eventBuilderById.containsKey(id)) {
                eventBuilderById.put(id, QuotaHistoryEvent.builder()
                        .id(id)
                        .quotaId(quotaId)
                        .quotaKey(key)
                        .personId(getLong(rs, "person_id"))
                        .tvmId(getLong(rs, "tvm_id"))
                        .updated(rs.getTimestamp("updated").toInstant())
                        .comment(rs.getString("comment"))
                        .oldMax(rs.getLong("old_max"))
                        .newMax(rs.getLong("new_max"))
                        .oldOwnMax(rs.getLong("old_own_max"))
                        .newOwnMax(rs.getLong("new_own_max"))
                        .issueKey(rs.getString("issue_key"))
                );
            }
        }
        return eventBuilderById.values()
                .stream()
                .map(QuotaHistoryEvent.Builder::build)
                .collect(Collectors.toList());
    }
}
