package ru.yandex.qe.dispenser.domain.dao.history.request;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Timestamp;
import java.time.Instant;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.stream.Collectors;

import com.google.common.collect.ImmutableMap;
import org.postgresql.util.PGobject;
import org.springframework.beans.factory.annotation.Autowired;

import ru.yandex.qe.dispenser.api.v1.DiOrder;
import ru.yandex.qe.dispenser.api.v1.DiQuotaRequestHistoryEventType;
import ru.yandex.qe.dispenser.domain.QuotaChangeRequestHistoryEvent;
import ru.yandex.qe.dispenser.domain.dao.SqlDaoBase;
import ru.yandex.qe.dispenser.domain.dao.SqlUtils;
import ru.yandex.qe.dispenser.domain.dao.quota.request.QuotaChangeRequestReader;
import ru.yandex.qe.dispenser.domain.dao.quota.request.SqlQuotaChangeRequestDao;
import ru.yandex.qe.dispenser.domain.util.HistoryDaoUtils;
import ru.yandex.qe.dispenser.domain.util.RelativePage;
import ru.yandex.qe.dispenser.domain.util.RelativePageInfo;

import static ru.yandex.qe.dispenser.domain.QuotaChangeRequestHistoryEvent.DATA_TYPE;
import static ru.yandex.qe.dispenser.domain.util.HistoryDaoUtils.operatorByOrder;
import static ru.yandex.qe.dispenser.domain.util.HistoryDaoUtils.toCondition;

public class SqlQuotaChangeRequestHistoryDao extends SqlDaoBase implements QuotaChangeRequestHistoryDao {
    private static final String LIMIT = " LIMIT :pageSize";
    private static final String ORDER = " ORDER BY id %s";

    private static final String CREATE_QUERY = "INSERT INTO quota_request_history(quota_request_id, event_type, person_id, tvm_id, comment, updated, old_data, new_data) " +
            "VALUES(:quotaRequestId, cast(:eventType as quota_request_history_event_type), :personId, :tvmId, :comment, :updated, :oldData, :newData)";
    private static final String GET_BY_ID_QUERY = "SELECT * FROM quota_request_history WHERE id = :id";
    private static final String GET_BY_REQUEST_IDS_QUERY_TEMPLATE = "SELECT * FROM quota_request_history WHERE quota_request_id IN (%s) AND id %s :offset";
    private static final String CLEAR_QUERY = "TRUNCATE quota_request_history CASCADE";
    private static final String CLEAR_BEFORE_QUERY = "DELETE FROM quota_request_history WHERE updated < :moment";

    @Autowired
    private SqlQuotaChangeRequestDao sqlQuotaChangeRequestDao;

    @Override
    public QuotaChangeRequestHistoryEvent create(final QuotaChangeRequestHistoryEvent event) {
        final long id = jdbcTemplate.insert(CREATE_QUERY, toParams(event));
        event.setId(id);
        return event;
    }

    @Override
    public void create(final Collection<QuotaChangeRequestHistoryEvent> events) {
        if (events.isEmpty()) {
            return;
        }
        jdbcTemplate.batchUpdate(CREATE_QUERY, events.stream().map(SqlQuotaChangeRequestHistoryDao::toParams).collect(Collectors.toList()));
    }

    @Override
    public QuotaChangeRequestHistoryEvent read(final long id) {
        return jdbcTemplate.queryForObject(GET_BY_ID_QUERY, ImmutableMap.of("id", id), SqlQuotaChangeRequestHistoryDao::map);
    }

    @Override
    public RelativePage<QuotaChangeRequestHistoryEvent> read(final QuotaChangeRequestReader.QuotaChangeRequestFilter filter,
                                                             final QuotaChangeRequestHistoryFilter historyFilter,
                                                             final RelativePageInfo pageInfo,
                                                             final DiOrder order) {
        final Map<String, Object> params = new HashMap<>();
        final String subquery = sqlQuotaChangeRequestDao.queryForIds(filter, params);
        final String condition = toCondition(historyFilter, false);
        params.putAll(HistoryDaoUtils.toParams(historyFilter));
        return getRelativePage(String.format(GET_BY_REQUEST_IDS_QUERY_TEMPLATE, subquery, operatorByOrder(order)) + condition + orderStatement(order) + LIMIT, params,
                SqlQuotaChangeRequestHistoryDao::map, pageInfo);
    }

    @Override
    public RelativePage<QuotaChangeRequestHistoryEvent> readHistoryByRequest(final long requestId, final QuotaChangeRequestHistoryFilter historyFilter,
                                                                             final RelativePageInfo pageInfo, final DiOrder order) {
        final Map<String, Object> params = new HashMap<>(HistoryDaoUtils.toParams(historyFilter));
        final String condition = toCondition(historyFilter, false);
        params.put("requestIds", Collections.singleton(requestId));
        return getRelativePage(String.format(GET_BY_REQUEST_IDS_QUERY_TEMPLATE, requestId, operatorByOrder(order)) + condition + orderStatement(order) + LIMIT, params,
                SqlQuotaChangeRequestHistoryDao::map, pageInfo);
    }

    private static String orderStatement(final DiOrder order) {
        return String.format(ORDER, order.toString());
    }

    @Override
    public boolean clear() {
        return jdbcTemplate.update(CLEAR_QUERY) > 0;
    }

    @Override
    public boolean clearBefore(final Instant moment) {
        return jdbcTemplate.update(CLEAR_BEFORE_QUERY, ImmutableMap.of("moment", Timestamp.from(moment))) > 0;
    }

    private static Map<String, ?> toParams(final QuotaChangeRequestHistoryEvent event) {
        final Map<String, Object> params = new HashMap<>();
        params.put("quotaRequestId", event.getQuotaChangeRequestId());
        params.put("eventType", event.getType().toString());
        params.put("personId", event.getPersonId());
        params.put("tvmId", event.getTvmId());
        params.put("comment", event.getComment());
        params.put("updated", Timestamp.from(event.getUpdated()));
        params.put("oldData", SqlUtils.toJsonb(event.getOldData()));
        params.put("newData", SqlUtils.toJsonb(event.getNewData()));
        return params;
    }

    private static QuotaChangeRequestHistoryEvent map(final ResultSet rs, final int i) throws SQLException {
        final QuotaChangeRequestHistoryEvent event = new QuotaChangeRequestHistoryEvent(
                getLong(rs, "person_id"),
                getLong(rs, "tvm_id"),
                rs.getTimestamp("updated").toInstant(),
                rs.getString("comment"),
                rs.getLong("quota_request_id"),
                DiQuotaRequestHistoryEventType.valueOf(rs.getString("event_type")),
                SqlUtils.fromJsonb((PGobject) rs.getObject("old_data"), DATA_TYPE),
                SqlUtils.fromJsonb((PGobject) rs.getObject("new_data"), DATA_TYPE)
        );
        event.setId(rs.getLong("id"));
        return event;
    }
}
